#include "../dat/DAT_ARRAY_CONSTS.hh"
#include "../dat/DAT_DISULFIDE_LINKS.hh"
#include "../dat/DAT_ENERGY_PARAMS.hh"
#include "../dat/DAT_PHYSICS_CONSTS.hh"
#include "../dat/DAT_REGION_MAPS.hh"
#include "../dat/DAT_RESIDUE_MAPPINGS.hh"
#include "../fil/Search_Subspace.hh"
#include "../fil/Structure.hh"
#include "../fil/Trajectory.hh"
#include "../phi/Coordinates.hh"
#include "../str/Ionstate_Automatic.hh"
#include "../str/Output_Streams.hh"
#include "../str/Thread_Options.hh"
#include <string>
#include <iostream>
#include <cstdlib>
#include <fstream>
#include <sstream>
#include <iomanip>
#include <cmath>

int main(int argc, char* argv[]){

   if( argc!=3 ){
      std::cerr<<"ERROR: Number of command line arguments "
                 "should equal 2.\n";
      std::exit( 1);
   }
   std::string FAMILY=argv[1];
   std::string PROTEIN=argv[2];

   std::string filename="../../"+FAMILY+"/exp/"+PROTEIN+".pdb";
   std::ifstream file_exists(filename.c_str());
   if( !file_exists ){
      std::cerr<<"ERROR: File "+filename.erase( 0, 6)+" does not exist.\n";
      std::exit( 1);
   }else{
      file_exists.close();
   }

   DAT_PHYSICS_CONSTS physics_consts;
   DAT_ARRAY_CONSTS array_consts;
   DAT_DISULFIDE_LINKS disulfide_links;
   DAT_ENERGY_PARAMS energy_params(physics_consts);
   DAT_RESIDUE_MAPPINGS residue_mappings(physics_consts,array_consts,
                                         energy_params);
   DAT_REGION_MAPS region_maps;

   Thread_Options opt;
   {
      opt.MODE="lp ";
      opt.Rcut= (999.00);
      opt.iW0_active=2;
   }
   Output_Streams out;
   {
//    filename="../../"+FAMILY+"/dgn/ionstate_bod."+PROTEIN;
//    out.FILE1.open(filename.c_str());
      filename="../../"+FAMILY+"/dgn/ionstate_titration."+PROTEIN;
      out.FILE2.open(filename.c_str());
      filename="../../"+FAMILY+"/dgn/ionstate."+PROTEIN;
      out.FILE3.open(filename.c_str());
//    out.FILE1<< std::fixed<< std::setprecision( 2);
      out.FILE2<< std::fixed<< std::setprecision( 2);
      out.FILE3<< std::fixed<< std::setprecision( 2);
      out.FAMILY=FAMILY;
      out.PROTEIN=PROTEIN;
      out.CONFORMATION="ph";
      out.VERBOSE=false;
   }

   Structure str;
   {
      str.fam=FAMILY;
      str.mol=PROTEIN;
      str.cnf="ph";
   }
   Trajectory tra;

   str.REFORM(physics_consts,residue_mappings);
   str.TOR(physics_consts,residue_mappings);
   tra.STR2TRA( 0,str);

// out.FILE3<<"REGULARIZE GEOMETRY"<< std::endl;
   str.REG(physics_consts,residue_mappings,out);
   str.POLARH(physics_consts,residue_mappings);
   str.TOR(physics_consts,residue_mappings);
   str.CAR(physics_consts,residue_mappings);
   str.SUP(physics_consts,out,tra);
   tra.STR2TRA( 1,str);
//
//
// interface to optimized params
//
   double a,b,c,d;                      //coeffs of Fpp, Fh, Fps, Fss
   a= energy_params.ion( 0)/
      energy_params.fac_pp;
   b= ( 1.00)/
      energy_params.fac_h;
   c= energy_params.ion( 1)/
      energy_params.fac_pp;
   d= energy_params.ion( 2)/
      energy_params.fac_pp;
   double ploc= energy_params.ion( 3);  //ddF= (1/x)*ln( 1 +x*ddF)
   double pglo= energy_params.ion( 4);  //hbond to unprotonated group
   int oT= 9;
   Ionstate_Automatic aut(oT);
   aut.T[ 0].g= energy_params.ion( 5);  //eala
   aut.T[ 1].g= energy_params.ion( 6);  //asp
   aut.T[ 2].g= energy_params.ion( 7);  //cyh
   aut.T[ 3].g= energy_params.ion( 8);  //glu
   aut.T[ 4].g= energy_params.ion( 9);  //his
   aut.T[ 5].g= energy_params.ion(10);  //lys
   aut.T[ 6].g= energy_params.ion(11);  //tyr
   aut.T[ 7].g= energy_params.ion(12);  //alae
   aut.T[ 8].g= energy_params.ion(13);  //hie
   {
      double dpK=(  0.00);
      double dg= (2.302585)*physics_consts.ekT*dpK;
      aut.T[ 0].g-=(.50)*dg;
      aut.T[ 1].g+=dg;
      aut.T[ 2].g+=dg;
      aut.T[ 3].g+=dg;
      aut.T[ 4].g-=dg;
      aut.T[ 5].g-=dg;
      aut.T[ 6].g+=dg;
      aut.T[ 7].g+=(.50)*dg;
      aut.T[ 8].g-=dg;
   }
   double pchg= energy_params.ion(14);  //counterion distribution
   aut.T[ 0].f= energy_params.ion(15);  //eala
   aut.T[ 1].f= energy_params.ion(16);  //asp
   aut.T[ 2].f= energy_params.ion(17);  //cyh
   aut.T[ 3].f= energy_params.ion(18);  //glu
   aut.T[ 4].f= energy_params.ion(19);  //his
   aut.T[ 5].f= energy_params.ion(20);  //lys
   aut.T[ 6].f= energy_params.ion(21);  //tyr
   aut.T[ 7].f= energy_params.ion(22);  //alae
   aut.T[ 8].f= energy_params.ion(23);  //hie
//
//
// input locked ionizable groups
//
   filename="../../"+FAMILY+"/stp/locked."+PROTEIN;
   std::ifstream ifile(filename.c_str());
   if( ifile ){
      std::string rec;
      int iT=-1;
      while( !std::getline(ifile,rec).eof() ){
         if( rec[ 0]=='#' ){
            iT++;
         }else{
            int iR0;
            std::istringstream(rec.substr( 0, 4))>>iR0;
            aut.T[iT].L.push_back( Ionstate_Automatic::tL());
            aut.T[iT].L[aut.T[iT].nL++].R0=(iR0-1);
         }
      }
      ifile.close();
      std::cout<<filename.erase( 0, 6)<< std::endl;
   }
//
// ionizable groups
// V0[].f should be calculated using consistent (a,b,c,d)
//
   double dpK=(  0.00);
   int oZ0=str.nZ0;
   int oV0=0;
   for(int iZ0=0;iZ0<oZ0;iZ0++){
      int mR0=str.Z0[iZ0].R0a;
      int nR0=(mR0-1+str.Z0[iZ0].cR0);
      for(int iR0=mR0;iR0<(mR0+1);iR0++){
         char c1=str.R0[iR0].c1;
         std::string aa=str.R0[iR0].aa;
         if      ( c1=='a' ){
            if( (aa[0]!='e')&&(aa[0]!='z') )continue;
            bool LOCKED=false;
            int oL=aut.T[ 0].L.size();
            for(int iL=0;iL<oL;iL++){
               if( aut.T[ 0].L[iL].R0==iR0 )LOCKED=true;
            }
            if( !LOCKED ){
               str.V0.push_back( Structure::tV0());
               str.V0[oV0].PROTONATED=( aa[0]=='z' )? false: true;
               str.V0[oV0].IONIZED=( aa[0]=='z' )? false: true;
               str.V0[oV0].T=0;
               str.V0[oV0].g= (2.302585)*physics_consts.ekT
                             *( ( 7.50) -(.5)*dpK);
               str.V0[oV0].f= aut.T[ 0].f;
               str.V0[oV0++].R0=iR0;
            }
            aa[0]='e';
            if      ( (aa=="eASP")||(aa=="eASZ") ){
               LOCKED=false;
               oL=aut.T[ 1].L.size();
               for(int iL=0;iL<oL;iL++){
                  if( aut.T[ 1].L[iL].R0==iR0 )LOCKED=true;
               }
               if( LOCKED )continue;
               str.V0.push_back( Structure::tV0());
               str.V0[oV0].PROTONATED=( aa=="eASZ" )? true: false;
               str.V0[oV0].IONIZED=( aa=="eASZ" )? false: true;
               str.V0[oV0].T=1;
               str.V0[oV0].g= (2.302585)*physics_consts.ekT*( ( 4.00) +dpK);
               str.V0[oV0].f= aut.T[ 1].f;
               str.V0[oV0++].R0=iR0;
            }else if( (aa=="eCYH")||(aa=="eCYZ") ){
               LOCKED=false;
               oL=aut.T[ 2].L.size();
               for(int iL=0;iL<oL;iL++){
                  if( aut.T[ 2].L[iL].R0==iR0 )LOCKED=true;
               }
               if( LOCKED )continue;
               str.V0.push_back( Structure::tV0());
               str.V0[oV0].PROTONATED=( aa=="eCYZ" )? false: true;
               str.V0[oV0].IONIZED=( aa=="eCYZ" )? true: false;
               str.V0[oV0].T=2;
               str.V0[oV0].g= (2.302585)*physics_consts.ekT*( ( 8.28) +dpK);
               str.V0[oV0].f= aut.T[ 2].f;
               str.V0[oV0++].R0=iR0;
            }else if( (aa=="eGLU")||(aa=="eGLZ") ){
               LOCKED=false;
               oL=aut.T[ 3].L.size();
               for(int iL=0;iL<oL;iL++){
                  if( aut.T[ 3].L[iL].R0==iR0 )LOCKED=true;
               }
               if( LOCKED )continue;
               str.V0.push_back( Structure::tV0());
               str.V0[oV0].PROTONATED=( aa=="eGLZ" )? true: false;
               str.V0[oV0].IONIZED=( aa=="eGLZ" )? false: true;
               str.V0[oV0].T=3;
               str.V0[oV0].g= (2.302585)*physics_consts.ekT*( ( 4.40) +dpK);
               str.V0[oV0].f= aut.T[ 3].f;
               str.V0[oV0++].R0=iR0;
            }else if( (aa=="eHIS")||(aa=="eHIP") ){
               LOCKED=false;
               oL=aut.T[ 4].L.size();
               for(int iL=0;iL<oL;iL++){
                  if( aut.T[ 4].L[iL].R0==iR0 )LOCKED=true;
               }
               if( LOCKED )continue;
               str.V0.push_back( Structure::tV0());
               str.V0[oV0].PROTONATED=( aa=="eHIP" )? true: false;
               str.V0[oV0].IONIZED=( aa=="eHIP" )? true: false;
               str.V0[oV0].T=4;
               str.V0[oV0].g= (2.302585)*physics_consts.ekT*( ( 6.30) -dpK);
               str.V0[oV0].f= aut.T[ 4].f;
               str.V0[oV0++].R0=iR0;
            }else if( (aa=="eLYS")||(aa=="eLYZ") ){
               LOCKED=false;
               oL=aut.T[ 5].L.size();
               for(int iL=0;iL<oL;iL++){
                  if( aut.T[ 5].L[iL].R0==iR0 )LOCKED=true;
               }
               if( LOCKED )continue;
               str.V0.push_back( Structure::tV0());
               str.V0[oV0].PROTONATED=( aa=="eLYZ" )? false: true;
               str.V0[oV0].IONIZED=( aa=="eLYZ" )? false: true;
               str.V0[oV0].T=5;
               str.V0[oV0].g= (2.302585)*physics_consts.ekT*( (10.50) -dpK);
               str.V0[oV0].f= aut.T[ 5].f;
               str.V0[oV0++].R0=iR0;
            }else if( (aa=="eTYR")||(aa=="eTYZ") ){
               LOCKED=false;
               oL=aut.T[ 6].L.size();
               for(int iL=0;iL<oL;iL++){
                  if( aut.T[ 6].L[iL].R0==iR0 )LOCKED=true;
               }
               if( LOCKED )continue;
               str.V0.push_back( Structure::tV0());
               str.V0[oV0].PROTONATED=( aa=="eTYZ" )? false: true;
               str.V0[oV0].IONIZED=( aa=="eTYZ" )? true: false;
               str.V0[oV0].T=6;
               str.V0[oV0].g= (2.302585)*physics_consts.ekT*( ( 9.60) +dpK);
               str.V0[oV0].f= aut.T[ 6].f;
               str.V0[oV0++].R0=iR0;
            }else if( aa=="eHIE" ){
               LOCKED=false;
               oL=aut.T[ 8].L.size();
               for(int iL=0;iL<oL;iL++){
                  if( aut.T[ 8].L[iL].R0==iR0 )LOCKED=true;
               }
               if( LOCKED )continue;
               str.V0.push_back( Structure::tV0());
               str.V0[oV0].PROTONATED=false;
               str.V0[oV0].IONIZED=false;
               str.V0[oV0].T=8;
               str.V0[oV0].g= (2.302585)*physics_consts.ekT*( ( 6.30) -dpK);
               str.V0[oV0].f= aut.T[ 8].f;
               str.V0[oV0++].R0=iR0;
            }else if( aa=="eARG" ){
            }
         }else if( c1=='e' ){
         }else if( c1=='r' ){
         }else if( c1=='b' ){
         }else if( c1=='p' ){
         }else if( c1=='s' ){
         }
      }
      for(int iR0=(mR0+1);iR0<nR0;iR0++){
         char c1=str.R0[iR0].c1;
         std::string aa=str.R0[iR0].aa;
         if      ( c1=='a' ){
            if      ( (aa=="ASP ")||(aa=="ASZ ") ){
               bool LOCKED=false;
               int oL=aut.T[ 1].L.size();
               for(int iL=0;iL<oL;iL++){
                  if( aut.T[ 1].L[iL].R0==iR0 )LOCKED=true;
               }
               if( LOCKED )continue;
               str.V0.push_back( Structure::tV0());
               str.V0[oV0].PROTONATED=( aa=="ASZ " )? true: false;
               str.V0[oV0].IONIZED=( aa=="ASZ " )? false: true;
               str.V0[oV0].T=1;
               str.V0[oV0].g= (2.302585)*physics_consts.ekT*( ( 4.00) +dpK);
               str.V0[oV0].f= aut.T[ 1].f;
               str.V0[oV0++].R0=iR0;
            }else if( (aa=="CYH ")||(aa=="CYZ ") ){
               bool LOCKED=false;
               int oL=aut.T[ 2].L.size();
               for(int iL=0;iL<oL;iL++){
                  if( aut.T[ 2].L[iL].R0==iR0 )LOCKED=true;
               }
               if( LOCKED )continue;
               str.V0.push_back( Structure::tV0());
               str.V0[oV0].PROTONATED=( aa=="CYZ " )? false: true;
               str.V0[oV0].IONIZED=( aa=="CYZ " )? true: false;
               str.V0[oV0].T=2;
               str.V0[oV0].g= (2.302585)*physics_consts.ekT*( ( 8.28) +dpK);
               str.V0[oV0].f= aut.T[ 2].f;
               str.V0[oV0++].R0=iR0;
            }else if( (aa=="GLU ")||(aa=="GLZ ") ){
               bool LOCKED=false;
               int oL=aut.T[ 3].L.size();
               for(int iL=0;iL<oL;iL++){
                  if( aut.T[ 3].L[iL].R0==iR0 )LOCKED=true;
               }
               if( LOCKED )continue;
               str.V0.push_back( Structure::tV0());
               str.V0[oV0].PROTONATED=( aa=="GLZ " )? true: false;
               str.V0[oV0].IONIZED=( aa=="GLZ " )? false: true;
               str.V0[oV0].T=3;
               str.V0[oV0].g= (2.302585)*physics_consts.ekT*( ( 4.40) +dpK);
               str.V0[oV0].f= aut.T[ 3].f;
               str.V0[oV0++].R0=iR0;
            }else if( (aa=="HIS ")||(aa=="HIP ") ){
               bool LOCKED=false;
               int oL=aut.T[ 4].L.size();
               for(int iL=0;iL<oL;iL++){
                  if( aut.T[ 4].L[iL].R0==iR0 )LOCKED=true;
               }
               if( LOCKED )continue;
               str.V0.push_back( Structure::tV0());
               str.V0[oV0].PROTONATED=( aa=="HIP " )? true: false;
               str.V0[oV0].IONIZED=( aa=="HIP " )? true: false;
               str.V0[oV0].T=4;
               str.V0[oV0].g= (2.302585)*physics_consts.ekT*( ( 6.30) -dpK);
               str.V0[oV0].f= aut.T[ 4].f;
               str.V0[oV0++].R0=iR0;
            }else if( (aa=="LYS ")||(aa=="LYZ ") ){
               bool LOCKED=false;
               int oL=aut.T[ 5].L.size();
               for(int iL=0;iL<oL;iL++){
                  if( aut.T[ 5].L[iL].R0==iR0 )LOCKED=true;
               }
               if( LOCKED )continue;
               str.V0.push_back( Structure::tV0());
               str.V0[oV0].PROTONATED=( aa=="LYZ " )? false: true;
               str.V0[oV0].IONIZED=( aa=="LYZ " )? false: true;
               str.V0[oV0].T=5;
               str.V0[oV0].g= (2.302585)*physics_consts.ekT*( (10.50) -dpK);
               str.V0[oV0].f= aut.T[ 5].f;
               str.V0[oV0++].R0=iR0;
            }else if( (aa=="TYR ")||(aa=="TYZ ") ){
               bool LOCKED=false;
               int oL=aut.T[ 6].L.size();
               for(int iL=0;iL<oL;iL++){
                  if( aut.T[ 6].L[iL].R0==iR0 )LOCKED=true;
               }
               if( LOCKED )continue;
               str.V0.push_back( Structure::tV0());
               str.V0[oV0].PROTONATED=( aa=="TYZ " )? false: true;
               str.V0[oV0].IONIZED=( aa=="TYZ " )? true: false;
               str.V0[oV0].T=6;
               str.V0[oV0].g= (2.302585)*physics_consts.ekT*( ( 9.60) +dpK);
               str.V0[oV0].f= aut.T[ 6].f;
               str.V0[oV0++].R0=iR0;
            }else if( aa=="HIE " ){
               bool LOCKED=false;
               int oL=aut.T[ 8].L.size();
               for(int iL=0;iL<oL;iL++){
                  if( aut.T[ 8].L[iL].R0==iR0 )LOCKED=true;
               }
               if( LOCKED )continue;
               str.V0.push_back( Structure::tV0());
               str.V0[oV0].PROTONATED=false;
               str.V0[oV0].IONIZED=false;
               str.V0[oV0].T=8;
               str.V0[oV0].g= (2.302585)*physics_consts.ekT*( ( 6.30) -dpK);
               str.V0[oV0].f= aut.T[ 8].f;
               str.V0[oV0++].R0=iR0;
            }else if( aa=="ARG " ){
            }
         }else if( c1=='e' ){
         }else if( c1=='r' ){
         }else if( c1=='b' ){
         }else if( c1=='p' ){
         }else if( c1=='s' ){
         }
      }
      for(int iR0=nR0;iR0<(nR0+1);iR0++){
         char c1=str.R0[iR0].c1;
         std::string aa=str.R0[iR0].aa;
         if      ( c1=='a' ){
            if( (aa[3]!='e')&&(aa[3]!='z') )continue;
            bool LOCKED=false;
            int oL=aut.T[ 7].L.size();
            for(int iL=0;iL<oL;iL++){
               if( aut.T[ 7].L[iL].R0==iR0 )LOCKED=true;
            }
            if( !LOCKED ){
               str.V0.push_back( Structure::tV0());
               str.V0[oV0].PROTONATED=( aa[3]=='z' )? true: false;
               str.V0[oV0].IONIZED=( aa[3]=='z' )? false: true;
               str.V0[oV0].T=7;
               str.V0[oV0].g= (2.302585)*physics_consts.ekT
                             *( ( 3.80) +(.5)*dpK);
               str.V0[oV0].f= aut.T[ 7].f;
               str.V0[oV0++].R0=iR0;
            }
            aa[3]='z';
            if      ( (aa=="ASPz")||(aa=="ASZz") ){
               LOCKED=false;
               oL=aut.T[ 1].L.size();
               for(int iL=0;iL<oL;iL++){
                  if( aut.T[ 1].L[iL].R0==iR0 )LOCKED=true;
               }
               if( LOCKED )continue;
               str.V0.push_back( Structure::tV0());
               str.V0[oV0].PROTONATED=( aa=="ASZz" )? true: false;
               str.V0[oV0].IONIZED=( aa=="ASZz" )? false: true;
               str.V0[oV0].T=1;
               str.V0[oV0].g= (2.302585)*physics_consts.ekT*( ( 4.00) +dpK);
               str.V0[oV0].f= aut.T[ 1].f;
               str.V0[oV0++].R0=iR0;
            }else if( (aa=="CYHz")||(aa=="CYZz") ){
               LOCKED=false;
               oL=aut.T[ 2].L.size();
               for(int iL=0;iL<oL;iL++){
                  if( aut.T[ 2].L[iL].R0==iR0 )LOCKED=true;
               }
               if( LOCKED )continue;
               str.V0.push_back( Structure::tV0());
               str.V0[oV0].PROTONATED=( aa=="CYZz" )? false: true;
               str.V0[oV0].IONIZED=( aa=="CYZz" )? true: false;
               str.V0[oV0].T=2;
               str.V0[oV0].g= (2.302585)*physics_consts.ekT*( ( 8.28) +dpK);
               str.V0[oV0].f= aut.T[ 2].f;
               str.V0[oV0++].R0=iR0;
            }else if( (aa=="GLUz")||(aa=="GLZz") ){
               LOCKED=false;
               oL=aut.T[ 3].L.size();
               for(int iL=0;iL<oL;iL++){
                  if( aut.T[ 3].L[iL].R0==iR0 )LOCKED=true;
               }
               if( LOCKED )continue;
               str.V0.push_back( Structure::tV0());
               str.V0[oV0].PROTONATED=( aa=="GLZz" )? true: false;
               str.V0[oV0].IONIZED=( aa=="GLZz" )? false: true;
               str.V0[oV0].T=3;
               str.V0[oV0].g= (2.302585)*physics_consts.ekT*( ( 4.40) +dpK);
               str.V0[oV0].f= aut.T[ 3].f;
               str.V0[oV0++].R0=iR0;
            }else if( (aa=="HISz")||(aa=="HIPz") ){
               LOCKED=false;
               oL=aut.T[ 4].L.size();
               for(int iL=0;iL<oL;iL++){
                  if( aut.T[ 4].L[iL].R0==iR0 )LOCKED=true;
               }
               if( LOCKED )continue;
               str.V0.push_back( Structure::tV0());
               str.V0[oV0].PROTONATED=( aa=="HIPz" )? true: false;
               str.V0[oV0].IONIZED=( aa=="HIPz" )? true: false;
               str.V0[oV0].T=4;
               str.V0[oV0].g= (2.302585)*physics_consts.ekT*( ( 6.30) -dpK);
               str.V0[oV0].f= aut.T[ 4].f;
               str.V0[oV0++].R0=iR0;
            }else if( (aa=="LYSz")||(aa=="LYZz") ){
               LOCKED=false;
               oL=aut.T[ 5].L.size();
               for(int iL=0;iL<oL;iL++){
                  if( aut.T[ 5].L[iL].R0==iR0 )LOCKED=true;
               }
               if( LOCKED )continue;
               str.V0.push_back( Structure::tV0());
               str.V0[oV0].PROTONATED=( aa=="LYZz" )? false: true;
               str.V0[oV0].IONIZED=( aa=="LYZz" )? false: true;
               str.V0[oV0].T=5;
               str.V0[oV0].g= (2.302585)*physics_consts.ekT*( (10.50) -dpK);
               str.V0[oV0].f= aut.T[ 5].f;
               str.V0[oV0++].R0=iR0;
            }else if( (aa=="TYRz")||(aa=="TYZz") ){
               LOCKED=false;
               oL=aut.T[ 6].L.size();
               for(int iL=0;iL<oL;iL++){
                  if( aut.T[ 6].L[iL].R0==iR0 )LOCKED=true;
               }
               if( LOCKED )continue;
               str.V0.push_back( Structure::tV0());
               str.V0[oV0].PROTONATED=( aa=="TYZz" )? false: true;
               str.V0[oV0].IONIZED=( aa=="TYZz" )? true: false;
               str.V0[oV0].T=6;
               str.V0[oV0].g= (2.302585)*physics_consts.ekT*( ( 9.60) +dpK);
               str.V0[oV0].f= aut.T[ 6].f;
               str.V0[oV0++].R0=iR0;
            }else if( aa=="HIEz" ){
               LOCKED=false;
               oL=aut.T[ 8].L.size();
               for(int iL=0;iL<oL;iL++){
                  if( aut.T[ 8].L[iL].R0==iR0 )LOCKED=true;
               }
               if( LOCKED )continue;
               str.V0.push_back( Structure::tV0());
               str.V0[oV0].PROTONATED=false;
               str.V0[oV0].IONIZED=false;
               str.V0[oV0].T=8;
               str.V0[oV0].g= (2.302585)*physics_consts.ekT*( ( 6.30) -dpK);
               str.V0[oV0].f= aut.T[ 8].f;
               str.V0[oV0++].R0=iR0;
            }else if( aa=="ARGz" ){
            }
         }else if( c1=='e' ){
         }else if( c1=='r' ){
         }else if( c1=='b' ){
         }else if( c1=='p' ){
         }else if( c1=='s' ){
         }
      }
   }
   str.nV0=oV0;
   if( oV0< 2 ){
      std::cerr<<"ERROR: Number of ionizable groups should be "
                 "greater than 1.\n";
      std::exit( 1);
   }
   std::cout<< std::fixed<< std::setprecision( 1);
   std::cout<<" ionizable groups"<< std::endl;
   for(int jV0=0;jV0<oV0;jV0++){
      int iR0=str.V0[jV0].R0;
      std::cout<<" res="<< std::setw( 4)<<(iR0+1)<<' '
               <<str.R0[iR0].aa<< std::endl;
   }
//
//
// hydrogen bond contacts to atom being protonated
//
   for(int jV0=0;jV0<oV0;jV0++){
      str.V0[jV0].hbc= (0.00);
      if( str.V0[jV0].PROTONATED )continue;
      int jR0=str.V0[jV0].R0;
      int iT=str.V0[jV0].T;
      std::string atm;
      if      ( iT==1 ){
         atm=" OD2";
      }else if( iT==3 ){
         atm=" OE2";
      }else if( iT==4 ){
         atm=" NE2";
      }else if( iT==7 ){
         atm=" OXT";
      }else if( iT==8 ){
         atm=" ND1";
      }else{
         continue;
      }
      int aP1=str.R0[jR0].P1a;
      int bP1=(aP1-1+str.R0[jR0].cP1);
      int jP1=-1;
      for(int kP1=aP1;kP1<=bP1&&(jP1==-1);kP1++){
         if( str.P1[kP1].sub==0 )continue;
         if( str.P1[kP1].atm==atm )jP1=kP1;
      }
      if( jP1==-1 )continue;
      for(int iZ0= 0;iZ0<oZ0;iZ0++){
         int mR0=str.Z0[iZ0].R0a;
         int nR0=(mR0-1+str.Z0[iZ0].cR0);
         for(int iR0=mR0;iR0<=nR0;iR0++){
            int mP1=str.R0[iR0].P1a;
            int nP1=(mP1-1+str.R0[iR0].cP1);
            for(int iP1=mP1;iP1<=nP1;iP1++){
               if( str.P1[iP1].sub==0 )continue;
               if( (str.P1[iP1].typ!= 2)&&(str.P1[iP1].typ!= 5) )continue;
               Coordinates u;
               for(int kP1=mP1;kP1<=nP1;kP1++){
                  if( kP1==iP1 )continue;
                  if( str.P1[kP1].sub==0 )continue;
                  Coordinates d=( str.P1[iP1].x -str.P1[kP1].x);
                  if( d.r()<( 1.11) ){
                     u=d.normalize();
                     break;
                  }
               }
               Coordinates d=( str.P1[jP1].x -str.P1[iP1].x);
               double r= d.r();
               if( r>( 2.50) )continue;
               d/=r;
               double Cthet= dot(d,u);
               double thet= std::acos( Cthet)/physics_consts.RAD;
               if( thet>(  60.00) )continue;
               str.V0[jV0].hbc= (1.00);
            }
         }
      }
   }
//
//
// chg-chg repulsion at distance characteristic of protein
//
   int oR0=(str.Z0[oZ0-1].R0a+str.Z0[oZ0-1].cR0);
   double rchar= (2.4814)*std::exp( std::log( double(oR0))/(3.));
   double ee= pchg*physics_consts.CAL*physics_consts.ANG/rchar;
//
//
// total chg of fully protonated state
//
   int qfull=0;
   for(int iZ0=0;iZ0<oZ0;iZ0++){
      int mR0=str.Z0[iZ0].R0a;
      int nR0=(mR0-1+str.Z0[iZ0].cR0);
      for(int iR0=mR0;iR0<(mR0+1);iR0++){
         char c1=str.R0[iR0].c1;
         std::string aa=str.R0[iR0].aa;
         if      ( c1=='a' ){
            if( (aa[0]!='e')&&(aa[0]!='z') )continue;
            aa[0]='e';
            if( aa=="eARG" ){
               qfull++;
            }
         }else if( c1=='e' ){
         }else if( c1=='r' ){
         }else if( c1=='b' ){
         }else if( c1=='p' ){
         }else if( c1=='s' ){
         }
      }
      for(int iR0=(mR0+1);iR0<nR0;iR0++){
         char c1=str.R0[iR0].c1;
         std::string aa=str.R0[iR0].aa;
         if      ( c1=='a' ){
            if( aa=="ARG " ){
               qfull++;
            }
         }else if( c1=='e' ){
         }else if( c1=='r' ){
         }else if( c1=='b' ){
         }else if( c1=='p' ){
         }else if( c1=='s' ){
         }
      }
      for(int iR0=nR0;iR0<(nR0+1);iR0++){
         char c1=str.R0[iR0].c1;
         std::string aa=str.R0[iR0].aa;
         if      ( c1=='a' ){
            if( (aa[3]!='e')&&(aa[3]!='z') )continue;
            aa[3]='e';
            if( aa=="ARGe" ){
               qfull++;
            }
         }else if( c1=='e' ){
         }else if( c1=='r' ){
         }else if( c1=='b' ){
         }else if( c1=='p' ){
         }else if( c1=='s' ){
         }
      }
   }
   for(int iT= 0;iT<oT;iT++){
      int oL=aut.T[iT].nL;
      for(int iL= 0;iL<oL;iL++){
         int iR0=aut.T[iT].L[iL].R0;
         std::string aa=str.R0[iR0].aa;
         std::string ab=aa;
         if      ( (aa[0]=='e')||(aa[0]=='z') ){
            ab=aa.substr(1,3)+' ';
         }else if( (aa[3]=='e')||(aa[3]=='z') ){
            ab=aa.substr(0,3)+' ';
         }
         if      ( iT==0 ){
            if( aa[0]=='e' )qfull++;
         }else if( iT==1 ){
            if( ab=="ASP " )qfull--;
         }else if( iT==2 ){
            if( ab=="CYZ " )qfull--;
         }else if( iT==3 ){
            if( ab=="GLU " )qfull--;
         }else if( iT==4 ){
            if( ab=="HIP " )qfull++;
         }else if( iT==5 ){
            if( ab=="LYS " )qfull++;
         }else if( iT==6 ){
            if( ab=="TYZ " )qfull--;
         }else if( iT==7 ){
            if( aa[3]=='e' )qfull--;
         }else if( iT==8 ){
            if( ab=="HIP " )qfull++;
         }
      }
   }
   for(int jV0= 0;jV0<oV0;jV0++){
      int jT=str.V0[jV0].T;
      if      ( jT== 0 ){
         qfull++;
      }else if( jT== 1 ){
      }else if( jT== 2 ){
      }else if( jT== 3 ){
      }else if( jT== 4 ){
         qfull++;
      }else if( jT== 5 ){
         qfull++;
      }else if( jT== 6 ){
      }else if( jT== 7 ){
      }else if( jT== 8 ){
         qfull++;
      }
   }
//
//
// fully protonated reference state
//
   int nI8=0;                           //number of unique ionization states
   aut.o_V0N2.resize(oV0, 0);
   double f= (0.00);                    //F of fully protonated reference state
   {
      out.VERBOSE=false;
      bool REDUCE=false;
      aut.I8.push_back( Ionstate_Automatic::tI8(oV0));
      aut.populate(aut.I8[nI8].sub,aut.I8[nI8].str,
                   physics_consts,residue_mappings,region_maps,
                   out,
                   str,REDUCE);
      double Fr,Fe,Ft,Fb,Fh, Fps,Fss;
      aut.evaluate(Fr,Fe,Ft,Fb,Fh, Fps,Fss,
                   physics_consts,array_consts,disulfide_links,
                   energy_params,residue_mappings,region_maps,
                   opt,out,
                   aut.I8[nI8].sub,aut.I8[nI8].str);
      f=( a*( Fr +Fe +Ft +Fb) +b*Fh +c*Fps +d*Fss);
      for(int jV0= 0;jV0<oV0;jV0++){
         aut.I8[nI8].V0N2(jV0)=aut.V0N2(jV0);
      }
      aut.I8[nI8].Fpp=( Fr +Fe +Ft +Fb);
      aut.I8[nI8].Fh= Fh;
      aut.I8[nI8].Fps= Fps;
      aut.I8[nI8].Fss= Fss;
      nI8++;
      out.VERBOSE=false;
   }
//
//
// ionization state of initial structure
//
   for(int jV0= 0;jV0<oV0;jV0++){
      int jT=str.V0[jV0].T;
      std::string aa=str.R0[str.V0[jV0].R0].aa;
      aut.V0N2(jV0)=0;
      if      ( jT== 0 ){
         if( aa[0]=='z' ){
           aut.V0N2(jV0)=1;
         }else{
         }
      }else if( jT== 1 ){
         if( aa=="ASZ " ){
         }else{
           aut.V0N2(jV0)=1;
         }
      }else if( jT== 2 ){
         if( aa=="CYZ " ){
           aut.V0N2(jV0)=1;
         }else{
         }
      }else if( jT== 3 ){
         if( aa=="GLZ " ){
         }else{
           aut.V0N2(jV0)=1;
         }
      }else if( jT== 4 ){
         if( aa=="HIP " ){
         }else{
           aut.V0N2(jV0)=1;
         }
      }else if( jT== 5 ){
         if( aa=="LYS " ){
         }else{
           aut.V0N2(jV0)=1;
         }
      }else if( jT== 6 ){
         if( aa=="TYR " ){
         }else{
           aut.V0N2(jV0)=1;
         }
      }else if( jT== 7 ){
         if( aa[3]=='z' ){
         }else{
           aut.V0N2(jV0)=1;
         }
      }else if( jT== 8 ){
         if( aa=="HIP " ){
         }else{
           aut.V0N2(jV0)=1;
         }
      }
   }
//
//
// initial structure reference state
//
   double f0= (0.00);                   //F of initial structure ref state
   {
      out.VERBOSE=false;
      bool REDUCE=false;
      aut.I8.push_back( Ionstate_Automatic::tI8(oV0));
      aut.populate(aut.I8[nI8].sub,aut.I8[nI8].str,
                   physics_consts,residue_mappings,region_maps,
                   out,
                   str,REDUCE);
      double Fr,Fe,Ft,Fb,Fh, Fps,Fss;
      aut.evaluate(Fr,Fe,Ft,Fb,Fh, Fps,Fss,
                   physics_consts,array_consts,disulfide_links,
                   energy_params,residue_mappings,region_maps,
                   opt,out,
                   aut.I8[nI8].sub,aut.I8[nI8].str);
      f0=( a*( Fr +Fe +Ft +Fb) +b*Fh +c*Fps +d*Fss);
      for(int jV0= 0;jV0<oV0;jV0++){
         aut.I8[nI8].V0N2(jV0)=aut.V0N2(jV0);
      }
      aut.I8[nI8].Fpp=( Fr +Fe +Ft +Fb);
      aut.I8[nI8].Fh= Fh;
      aut.I8[nI8].Fps= Fps;
      aut.I8[nI8].Fss= Fss;
      nI8++;
      out.VERBOSE=false;
   }
   double fref0= (0.00);
   int q0=qfull;
   for(int jV0= 0;jV0<oV0;jV0++){
      if( aut.V0N2(jV0)==1 ){
         fref0+=str.V0[jV0].f;
         fref0+=aut.T[str.V0[jV0].T].g;
         q0--;
      }
   }
   double fee0= (0.00);
   if      ( q0<-1 ){
      fee0=-double( q0*(q0+1)/2)*ee;
   }else if( q0> 1 ){
      fee0=-double( q0*(q0-1)/2)*ee;
   }
//
//
// 1-bod deprotonation
//
   for(int jV0= 0;jV0<oV0;jV0++){
      aut.V0N2(jV0)=0;
   }
// out.FILE1<<"1-BOD DEPROTONATION"<< std::endl;
   out.VERBOSE=false;
   aut.o_zV0f.resize(oV0, (0.00));
   for(int pV0= 0;pV0<oV0;pV0++){
//    int pR0=str.V0[pV0].R0;
      bool REDUCE=true;
      aut.V0N2(pV0)=1;
      Search_Subspace msub;
      Structure mstr;
      aut.populate(msub,mstr,
                   physics_consts,residue_mappings,region_maps,
                   out,
                   aut.I8[ 0].str,REDUCE);
      double Fr,Fe,Ft,Fb,Fh, Fps,Fss;
      aut.evaluate(Fr,Fe,Ft,Fb,Fh, Fps,Fss,
                   physics_consts,array_consts,disulfide_links,
                   energy_params,residue_mappings,region_maps,
                   opt,out,
                   msub,mstr);
      aut.zV0f(pV0)+=( a*( Fr +Fe +Ft +Fb) +b*Fh +c*Fps +d*Fss);
//    out.FILE1<<"  "
//             <<"     F   "
//             <<"    Fr   "
//             <<"    Fe   "
//             <<"    Ft   "
//             <<"    Fb   "
//             <<"    Fh   "
//             <<"    Fm   "
//             <<"    Fps  "
//             <<"    Fss  "<< std::endl;
//    out.FILE1<<"D "
//             << std::setw(9)<<( a*( Fr +Fe +Ft +Fb) +b*Fh +c*Fps +d*Fss)
//             << std::setw(9)<<a*Fr
//             << std::setw(9)<<a*Fe
//             << std::setw(9)<<a*Ft
//             << std::setw(9)<<a*Fb
//             << std::setw(9)<<b*Fh
//             << std::setw(9)<<( c*Fps +d*Fss)
//             << std::setw(9)<<c*Fps
//             << std::setw(9)<<d*Fss<< std::endl;
      aut.V0N2(pV0)=0;
      Search_Subspace fsub;
      Structure fstr;
      aut.populate(fsub,fstr,
                   physics_consts,residue_mappings,region_maps,
                   out,
                   aut.I8[ 0].str,REDUCE);
      for(int iR1= 0;iR1<fsub.nR1;iR1++){
         fsub.R1[iR1].bb=msub.R1[iR1].bb;
         fsub.R1[iR1].sc=msub.R1[iR1].sc;
      }
      aut.evaluate(Fr,Fe,Ft,Fb,Fh, Fps,Fss,
                   physics_consts,array_consts,disulfide_links,
                   energy_params,residue_mappings,region_maps,
                   opt,out,
                   fsub,fstr);
      aut.zV0f(pV0)-=( a*( Fr +Fe +Ft +Fb) +b*Fh +c*Fps +d*Fss);
//    out.FILE1<<"DH"
//             << std::setw(9)<<( a*( Fr +Fe +Ft +Fb) +b*Fh +c*Fps +d*Fss)
//             << std::setw(9)<<a*Fr
//             << std::setw(9)<<a*Fe
//             << std::setw(9)<<a*Ft
//             << std::setw(9)<<a*Fb
//             << std::setw(9)<<b*Fh
//             << std::setw(9)<<( c*Fps +d*Fss)
//             << std::setw(9)<<c*Fps
//             << std::setw(9)<<d*Fss<< std::endl;
//    out.FILE1<<"[  res    ]"
//             <<"   dF  "
//             <<"   df  "
//             <<"  dF-df"<< std::endl;
//    out.FILE1<<'['<< std::setw( 4)<<(pR0+1)<<' '<<str.R0[pR0].aa<<']'
//             << std::setw( 7)<<aut.zV0f(pV0)
//             << std::setw( 7)<<str.V0[pV0].f
//             << std::setw( 7)<<( aut.zV0f(pV0) -str.V0[pV0].f)
//             << std::endl;
   }
   out.VERBOSE=false;
//
//
// 2-bod deprotonation
//
// out.FILE1<<"2-BOD DEPROTONATION"<< std::endl;
   out.VERBOSE=false;
   aut.o_V0=oV0;
   aut.o_zV0V0f.resize(oV0*oV0, (0.00));
   aut.o_zV0V0r.resize(oV0*oV0, (1.00e+6));
   for(int pV0= 0;pV0<(oV0-1);pV0++){
      int pR0=str.V0[pV0].R0;
      int pP1min=str.R0[pR0].P1a;
      int pP1max=(pP1min-1+str.R0[pR0].cP1);
      for(int qV0=(pV0+1);qV0<oV0;qV0++){
         int qR0=str.V0[qV0].R0;
         int qP1min=str.R0[qR0].P1a;
         int qP1max=(qP1min-1+str.R0[qR0].cP1);
         for(int pP1=pP1min;pP1<=pP1max;pP1++){
            for(int qP1=qP1min;qP1<=qP1max;qP1++){
               double r=( str.P1[qP1].x -str.P1[pP1].x).r();
               if( r<aut.zV0V0r(pV0,qV0) )aut.zV0V0r(pV0,qV0)= r;
            }
         }
         if( aut.zV0V0r(pV0,qV0)>( 32.00) )continue;
//       if( aut.zV0V0r(pV0,qV0)>( 24.00) )continue; //####
         bool REDUCE=true;
         aut.V0N2(pV0)=1;
         aut.V0N2(qV0)=1;
         Search_Subspace msub;
         Structure mstr;
         aut.populate(msub,mstr,
                      physics_consts,residue_mappings,region_maps,
                      out,
                      aut.I8[ 0].str,REDUCE);
         double Fr,Fe,Ft,Fb,Fh, Fps,Fss;
         aut.evaluate(Fr,Fe,Ft,Fb,Fh, Fps,Fss,
                      physics_consts,array_consts,disulfide_links,
                      energy_params,residue_mappings,region_maps,
                      opt,out,
                      msub,mstr);
         aut.zV0V0f(pV0,qV0)+=( a*( Fr +Fe +Ft +Fb) +b*Fh +c*Fps +d*Fss);
//       out.FILE1<<"  "
//                <<"     F   "
//                <<"    Fr   "
//                <<"    Fe   "
//                <<"    Ft   "
//                <<"    Fb   "
//                <<"    Fh   "
//                <<"    Fm   "
//                <<"    Fps  "
//                <<"    Fss  "<< std::endl;
//       out.FILE1<<"D "
//                << std::setw(9)<<( a*( Fr +Fe +Ft +Fb) +b*Fh +c*Fps +d*Fss)
//                << std::setw(9)<<a*Fr
//                << std::setw(9)<<a*Fe
//                << std::setw(9)<<a*Ft
//                << std::setw(9)<<a*Fb
//                << std::setw(9)<<b*Fh
//                << std::setw(9)<<( c*Fps +d*Fss)
//                << std::setw(9)<<c*Fps
//                << std::setw(9)<<d*Fss<< std::endl;
         aut.V0N2(pV0)=0;
         aut.V0N2(qV0)=0;
         Search_Subspace fsub;
         Structure fstr;
         aut.populate(fsub,fstr,
                      physics_consts,residue_mappings,region_maps,
                      out,
                      aut.I8[ 0].str,REDUCE);
         for(int iR1= 0;iR1<fsub.nR1;iR1++){
            fsub.R1[iR1].bb=msub.R1[iR1].bb;
            fsub.R1[iR1].sc=msub.R1[iR1].sc;
         }
         aut.evaluate(Fr,Fe,Ft,Fb,Fh, Fps,Fss,
                      physics_consts,array_consts,disulfide_links,
                      energy_params,residue_mappings,region_maps,
                      opt,out,
                      fsub,fstr);
         aut.zV0V0f(pV0,qV0)-=( a*( Fr +Fe +Ft +Fb) +b*Fh +c*Fps +d*Fss
                               +aut.zV0f(pV0)
                               +aut.zV0f(qV0));
//       out.FILE1<<"DH"
//                << std::setw(9)<<( a*( Fr +Fe +Ft +Fb) +b*Fh +c*Fps +d*Fss)
//                << std::setw(9)<<a*Fr
//                << std::setw(9)<<a*Fe
//                << std::setw(9)<<a*Ft
//                << std::setw(9)<<a*Fb
//                << std::setw(9)<<b*Fh
//                << std::setw(9)<<( c*Fps +d*Fss)
//                << std::setw(9)<<c*Fps
//                << std::setw(9)<<d*Fss<< std::endl;
//       out.FILE1<<"[  res    ]"
//                <<"[  res    ]"
//                <<"    r  "
//                <<"   dF  "<< std::endl;
//       out.FILE1<<'['<< std::setw( 4)<<(pR0+1)<<' '<<str.R0[pR0].aa<<']'
//                <<'['<< std::setw( 4)<<(qR0+1)<<' '<<str.R0[qR0].aa<<']'
//                << std::setw( 7)<<aut.zV0V0r(pV0,qV0)
//                << std::setw( 7)<<aut.zV0V0f(pV0,qV0)
//                << std::endl;
      }
   }
   out.VERBOSE=false;
//
//
// additional 1-bod pH independent
//
   for(int pV0= 0;pV0<oV0;pV0++){
      aut.zV0f(pV0)-=str.V0[pV0].f;
      aut.zV0f(pV0)-=aut.T[str.V0[pV0].T].g;
      aut.zV0f(pV0)+=(pglo*str.V0[pV0].hbc);
   }
//
//
// titration
//
   aut.o_V0N2f.resize(oV0*2);
   aut.o_V0V0f.resize(oV0*oV0);
   aut.o_V0N2V0f.resize(oV0*2*oV0);
   aut.o_V0N2V0N2f.resize(oV0*2*oV0*2);
   aut.o_V0g.resize(oV0);
   aut.o_V0h.resize(oV0);
   int oK8=512;
   aut.o_K8h.resize(oK8);
   aut.o_K8ord.resize(oK8);
   aut.o_K8inv.resize(oK8);
   int nH6=0;                //number of pH values
   for(double pH=(11.5);pH>( 0.0);pH-=( 0.1)){
      std::cout<<" pH="<< std::setw( 6)<<pH<< std::endl;
      aut.H6.push_back( Ionstate_Automatic::tH6(oV0,pH));
//
//
// pH dependent 2-bod interaction energy
//
      for(int pV0= 0;pV0<oV0;pV0++){
         aut.V0N2f(pV0, 0)= (0.00);
         aut.V0N2f(pV0, 1)= aut.zV0f(pV0);
         double dg=( str.V0[pV0].g -(2.302585)*physics_consts.ekT*pH);
         if( dg> (0.00) ){
            aut.V0N2f(pV0, 1)+=dg;
         }else{
            aut.V0N2f(pV0, 0)-=dg;
         }
      }
      double w= (1.00)/double(oV0-1);
      for(int pV0= 0;pV0<(oV0-1);pV0++){
         for(int qV0=(pV0+1);qV0<oV0;qV0++){
            aut.V0V0f(pV0,qV0)= (1.00e+8);
            for(int pN2=0;pN2<2;pN2++){
               aut.V0N2V0f(pV0,pN2,qV0)= (1.00e+8);
               for(int qN2=0;qN2<2;qN2++){
                  aut.V0N2V0N2f(pV0,pN2,qV0,qN2)= w*( aut.V0N2f(pV0,pN2)
                                                     +aut.V0N2f(qV0,qN2));
                  if( (pN2==1)&&(qN2==1) ){
                     aut.V0N2V0N2f(pV0,pN2,qV0,qN2)+=aut.zV0V0f(pV0,qV0);
                  }
                  if( aut.V0N2V0N2f(pV0,pN2,qV0,qN2)<aut.V0V0f(pV0,qV0) ){
                     aut.V0V0f(pV0,qV0)= aut.V0N2V0N2f(pV0,pN2,qV0,qN2);
                  }
                  if( aut.V0N2V0N2f(pV0,pN2,qV0,qN2)<aut.V0N2V0f(pV0,pN2,qV0) ){
                     aut.V0N2V0f(pV0,pN2,qV0)= aut.V0N2V0N2f(pV0,pN2,qV0,qN2);
                  }
               }
            }
         }
      }
      double h0= (0.00);
      for(int pV0= 0;pV0<(oV0-1);pV0++){
         for(int qV0=(pV0+1);qV0<oV0;qV0++){
            h0+=aut.V0V0f(pV0,qV0);
         }
      }
      double g0= (0.00);
//
//
// combinatorial search to obtain collection of low dG ionization states
//
      aut.K8.resize(oK8, Ionstate_Automatic::tH6::tH6K8(oV0));
      for(int jV0=0;jV0<oV0;jV0++){
         aut.V0N2(jV0)=-1;
      }
      int iV0= 0;
      for(;;){
         aut.V0N2(iV0)++;
         if( aut.V0N2(iV0)>1 ){
            if( iV0== 0 ){
               break;
            }else{
               aut.V0N2(iV0)=-1;
               iV0--;
               continue;
            }

         }else{
            aut.V0g(iV0)=( iV0> 0 )? aut.V0g(iV0-1): g0;
            aut.V0h(iV0)=( iV0> 0 )? aut.V0h(iV0-1): h0;
            for(int jV0= 0;jV0<iV0;jV0++){
               aut.V0g(iV0)+=aut.V0N2V0N2f(jV0,aut.V0N2(jV0),iV0,aut.V0N2(iV0));
               aut.V0h(iV0)-=aut.V0N2V0f(jV0,aut.V0N2(jV0),iV0);
            }
            for(int jV0=(iV0+1);jV0<oV0;jV0++){
               aut.V0h(iV0)-=aut.V0V0f(iV0,jV0);
               aut.V0h(iV0)+=aut.V0N2V0f(iV0,aut.V0N2(iV0),jV0);
            }
            if( (aut.V0g(iV0)+aut.V0h(iV0))>aut.K8[oK8-1].e )continue;
            if( iV0==(oV0-1) ){
               aut.K8[oK8-1].e= aut.V0g(iV0);
               for(int jV0= 0;jV0<oV0;jV0++){
                  aut.K8[oK8-1].V0N2(jV0)=aut.V0N2(jV0);
               }
               for(int iK8=(oK8-1);iK8> 0;iK8--){
                  if( aut.K8[iK8].e>=aut.K8[iK8-1].e )break;
                  double z= aut.K8[iK8-1].e;
                  aut.K8[iK8-1].e= aut.K8[iK8].e;
                  aut.K8[iK8].e= z;
                  for(int jV0= 0;jV0<oV0;jV0++){
                     int j=aut.K8[iK8-1].V0N2(jV0);
                     aut.K8[iK8-1].V0N2(jV0)=aut.K8[iK8  ].V0N2(jV0);
                     aut.K8[iK8  ].V0N2(jV0)=j;
                  }
               }
            }else{
               iV0++;
            }

         }
      }
      int nK8=oK8;
      while( (nK8>0)&&(aut.K8[nK8-1].e>=(1.00e+8)) ){
         nK8--;
      }
//
//
// dFmin=dF for path of initial structure reference state
//
      double e0= (0.00);
      for(int iV0= 1;iV0<oV0;iV0++){
         int iN2=aut.I8[ 1].V0N2(iV0);
         for(int jV0= 0;jV0<iV0;jV0++){
            int jN2=aut.I8[ 1].V0N2(jV0);
            e0+=aut.V0N2V0N2f(jV0,jN2,iV0,iN2);
         }
      }
      h0= (0.00);
      g0= (0.00);
      for(int iV0= 0;iV0<oV0;iV0++){
         int iN2=aut.I8[ 1].V0N2(iV0);
         double dg=( str.V0[iV0].g -(2.302585)*physics_consts.ekT*pH);
         if( iN2==1 ){
            if( dg> (0.00) )g0+=dg;
            h0+=(pglo*str.V0[iV0].hbc);
         }else{
            if( dg< (0.00) )g0-=dg;
         }
      }
      double dFmin=( e0 -g0 -h0);
//
//
// replace K8e with better estimate of dG
//
      for(int iK8= 0;iK8<nK8;iK8++){
         aut.K8h(iK8)= (0.00);
         for(int jV0= 0;jV0<oV0;jV0++){
            double dg=( str.V0[jV0].g -(2.302585)*physics_consts.ekT*pH);
            if( aut.K8[iK8  ].V0N2(jV0)==1 ){
               if( dg> (0.00) )aut.K8[iK8].g+=dg;
               aut.K8h(iK8)+=(pglo*str.V0[jV0].hbc);
            }else{
               if( dg< (0.00) )aut.K8[iK8].g-=dg;
            }
         }
         aut.K8[iK8].dF=( aut.K8[iK8].e -aut.K8[iK8].g
                         -aut.K8h(iK8));
      }
      for(int iK8= 0;iK8<nK8;iK8++){
         double ddF=( aut.K8[iK8].dF -dFmin);
         if( ddF> (0.00) ){
            aut.K8[iK8].e=( dFmin +((1.00)/ploc)*std::log( (1.00) +ploc*ddF)
                           +aut.K8[iK8].g +aut.K8h(iK8));
         }
      }
//
//
// reorder K8
//
      if( nK8> 1 ){
         for(int iK8= 0;iK8<nK8;iK8++){
            aut.K8ord(iK8)=iK8;
         }
         for(int iK8max=(nK8-1);iK8max>0;iK8max--){
            bool ORDERED=true;
            double e1= aut.K8[aut.K8ord( 0)].e;
            for(int iK8= 1;iK8<=iK8max;iK8++){
               double e2= aut.K8[aut.K8ord(iK8)].e;
               if( e2<e1 ){
                  int i=aut.K8ord(iK8);
                  aut.K8ord(iK8)=aut.K8ord(iK8-1);
                  aut.K8ord(iK8-1)=i;
                  ORDERED=false;
               }else{
                  e1= e2;
               }
            }
            if( ORDERED )break;
         }
         for(int iK8= 0;iK8<nK8;iK8++){
            aut.K8inv(aut.K8ord(iK8))=iK8;
         }
         for(int iK8= 0;iK8<nK8;iK8++){
            int jK8=aut.K8ord(iK8);
            int kK8=aut.K8inv(iK8);
            if( jK8==iK8 )continue;
            double z=aut.K8[iK8].e;
            aut.K8[iK8].e= aut.K8[jK8].e;
            aut.K8[jK8].e=z;
            for(int jV0= 0;jV0<oV0;jV0++){
               int i=aut.K8[iK8  ].V0N2(jV0);
               aut.K8[iK8  ].V0N2(jV0)=aut.K8[jK8  ].V0N2(jV0);
               aut.K8[jK8  ].V0N2(jV0)=i;
            }
            aut.K8ord(iK8)=iK8;
            aut.K8inv(iK8)=iK8;
            aut.K8ord(kK8)=jK8;
            aut.K8inv(jK8)=kK8;
         }
      }
//
//
// variable nK8 based on density of ion states
//
      nK8=256;
      double ebound=( aut.K8[ 0].e +( 8.00));
      while( (nK8>0)&&(aut.K8[nK8-1].e>ebound) ){
         nK8--;
      }
//
//
// (dF, g) for ionization states
//
      out.VERBOSE=false;
      aut.H6[nH6].nK8=nK8;
      for(int iK8= 0;iK8<nK8;iK8++){
         aut.H6[nH6].K8.push_back( Ionstate_Automatic::tH6::tH6K8(oV0));
         aut.H6[nH6].K8[iK8].e= aut.K8[iK8].e;
         for(int jV0= 0;jV0<oV0;jV0++){
            aut.H6[nH6].K8[iK8].V0N2(jV0)=aut.K8[iK8].V0N2(jV0);
            aut.V0N2(jV0)=aut.K8[iK8].V0N2(jV0);
         }
         double g= (0.00);
         double fref= (0.00);
         int q=qfull;
         for(int jV0= 0;jV0<oV0;jV0++){
            double dg=( str.V0[jV0].g -(2.302585)*physics_consts.ekT*pH);
            if( aut.V0N2(jV0)==1 ){
               if( dg> (0.00) )g+=dg;
               fref+=str.V0[jV0].f;
               fref+=aut.T[str.V0[jV0].T].g;
               q--;
            }else{
               if( dg< (0.00) )g-=dg;
            }
         }
         aut.H6[nH6].K8[iK8].g= g;
         double fee= (0.00);
         if      ( q<-1 ){
            fee=-double( q*(q+1)/2)*ee;
         }else if( q> 1 ){
            fee=-double( q*(q-1)/2)*ee;
         }
         int jI8=-1;
         for(int iI8= 0;iI8<nI8;iI8++){
            if( aut.I8[iI8].o_V0N2==aut.o_V0N2 ){
               jI8=iI8;
            }
         }
         if( jI8==-1 ){
            int zmin=0;
            for(int jV0= 0;jV0<oV0;jV0++){
               int iN2=( str.V0[jV0].PROTONATED )? 0: 1;
               if( iN2!=aut.V0N2(jV0) )zmin++;
            }
            for(int iI8= 0;iI8<nI8;iI8++){
               int z=0;
               for(int jV0= 0;jV0<oV0;jV0++){
                  if( aut.I8[iI8].V0N2(jV0)!=aut.V0N2(jV0) )z++;
               }
               if( z<zmin ){
                  zmin=z;
                  jI8=iI8;
               }
            }
            bool REDUCE=false;
            aut.I8.push_back( Ionstate_Automatic::tI8(oV0));
            if( jI8==-1 ){
               aut.populate(aut.I8[nI8].sub,aut.I8[nI8].str,
                            physics_consts,residue_mappings,region_maps,
                            out,
                            str,REDUCE);
            }else{
               aut.populate(aut.I8[nI8].sub,aut.I8[nI8].str,
                            physics_consts,residue_mappings,region_maps,
                            out,
                            aut.I8[jI8].str,REDUCE);
            }
            double Fr,Fe,Ft,Fb,Fh, Fps,Fss;
            aut.evaluate(Fr,Fe,Ft,Fb,Fh, Fps,Fss,
                         physics_consts,array_consts,disulfide_links,
                         energy_params,residue_mappings,region_maps,
                         opt,out,
                         aut.I8[nI8].sub,aut.I8[nI8].str);
            aut.H6[nH6].K8[iK8].dF=( a*( Fr +Fe +Ft +Fb) +b*Fh +c*Fps +d*Fss
                                    -f -fref +fee);
            for(int jV0= 0;jV0<oV0;jV0++){
               aut.I8[nI8].V0N2(jV0)=aut.V0N2(jV0);
            }
            aut.I8[nI8].Fpp=( Fr +Fe +Ft +Fb);
            aut.I8[nI8].Fh= Fh;
            aut.I8[nI8].Fps= Fps;
            aut.I8[nI8].Fss= Fss;
            aut.H6[nH6].K8[iK8].I8=nI8;
            nI8++;
            aut.H6[nH6].K8[iK8].pp=( Fr +Fe +Ft +Fb);
            aut.H6[nH6].K8[iK8].h= Fh;
            aut.H6[nH6].K8[iK8].ps= Fps;
            aut.H6[nH6].K8[iK8].ss= Fss;
            aut.H6[nH6].K8[iK8].EVAL=true;
         }else{
            aut.H6[nH6].K8[iK8].I8=jI8;
            aut.H6[nH6].K8[iK8].dF=( a*aut.I8[jI8].Fpp
                                    +b*aut.I8[jI8].Fh
                                    +c*aut.I8[jI8].Fps
                                    +d*aut.I8[jI8].Fss
                                    -f -fref +fee);
            aut.H6[nH6].K8[iK8].pp= aut.I8[jI8].Fpp;
            aut.H6[nH6].K8[iK8].h= aut.I8[jI8].Fh;
            aut.H6[nH6].K8[iK8].ps= aut.I8[jI8].Fps;
            aut.H6[nH6].K8[iK8].ss= aut.I8[jI8].Fss;
         }
      }
      aut.K8.clear();
      nH6++;
      out.VERBOSE=false;
   }
//
//
// dFmin=dF for initial structure reference state
//
   double dFmin=( f0 -f -fref0 +fee0);
//
//
// dG
//
   for(int iH6= 0;iH6<nH6;iH6++){
      int nK8=aut.H6[iH6].nK8;
      for(int iK8= 0;iK8<nK8;iK8++){
         double ddF=( aut.H6[iH6].K8[iK8].dF -dFmin);
         if( ddF> (0.00) ){
            aut.H6[iH6].K8[iK8].dG=( dFmin
                                    +((1.00)/ploc)*std::log( (1.00) +ploc*ddF)
                                    +aut.H6[iH6].K8[iK8].g);
         }else{
            aut.H6[iH6].K8[iK8].dG=( dFmin
                                    +ddF
                                    +aut.H6[iH6].K8[iK8].g);
         }
         for(int jV0= 0;jV0<oV0;jV0++){
            if( aut.H6[iH6].K8[iK8].V0N2(jV0)==1 ){
               aut.H6[iH6].K8[iK8].dG+=pglo*str.V0[jV0].hbc;
            }
         }
         if( aut.H6[iH6].K8[iK8].dG<aut.H6[iH6].dGmin ){
            aut.H6[iH6].jK8=iK8;
            aut.H6[iH6].dGmin= aut.H6[iH6].K8[iK8].dG;
         }
      }
//
//
// output structure(pH) with predicted ionization state
//
      int jI8=aut.H6[iH6].K8[aut.H6[iH6].jK8].I8;
      double pH=aut.H6[iH6].pH;
      char aph[3];
      int i=int(pH*10);
      int j=(i/100);
      aph[0]=char('0'+j);
      i-=(j*100);
      j=(i/10);
      aph[1]=char('0'+j);
      i-=(j*10);
      aph[2]=char('0'+i);
      std::string mol=aut.I8[jI8].str.mol;
      aut.I8[jI8].str.mol=mol+aph[0]+aph[1]+aph[2];
      aut.I8[jI8].str.SEQ2FIL();
      aut.I8[jI8].str.TOR2FIL(physics_consts);
      aut.I8[jI8].str.CAR2FIL();
      aut.I8[jI8].str.mol=mol;
   }
//
//
// pKa
//
   for(int iH6= 0;iH6<nH6;iH6++){
      int nK8=aut.H6[iH6].nK8;
      double z= (0.00);
      for(int iK8= 0;iK8<nK8;iK8++){
         double e=( aut.H6[iH6].K8[iK8].dG -aut.H6[iH6].dGmin);
         aut.H6[iH6].K8[iK8].p= std::exp( -e/physics_consts.ekT);
         z+=aut.H6[iH6].K8[iK8].p;
      }
      for(int iK8= 0;iK8<nK8;iK8++){
         aut.H6[iH6].K8[iK8].p/=z;
      }
      aut.H6[iH6].s=-physics_consts.ekT*std::log( z);
      for(int jV0= 0;jV0<oV0;jV0++){
         aut.H6[iH6].V0p(jV0)= (0.00);
         for(int iK8= 0;iK8<nK8;iK8++){
            if( aut.H6[iH6].K8[iK8].V0N2(jV0)==1 ){
               aut.H6[iH6].V0p(jV0)+=aut.H6[iH6].K8[iK8].p;
            }
         }
      }
   }
   for(int jV0= 0;jV0<oV0;jV0++){
      double zmin= (1.00e+6);
      str.V0[jV0].pKa= (-9.00);
      for(double pk=( -1.00);pk<(12.51);pk+=(  0.05)){
         double z= (0.00);
         for(int iH6= 0;iH6<nH6;iH6++){
            double frac= std::exp( ( aut.H6[iH6].pH -pk)*(2.302585));
            double p= (frac/( (1.00) +frac));
            double dp=( aut.H6[iH6].V0p(jV0) -p);
            z=z +dp*dp;
         }
         if( z<zmin ){
            zmin= z;
            str.V0[jV0].pKa= pk;
         }
      }
   }
//
//
// assemble header
//
   aut.o_V0a1.resize(oV0,' ');
   for(int jV0=0;jV0<oV0;jV0++){
      int iR0=str.V0[jV0].R0;
      int iL0=str.R0[iR0].L0;
      aut.V0a1(jV0)=residue_mappings.L0[iL0].a1;
   }
//
//
// write pKa of ionizable groups
//
   if( true ){
      out.FILE2<<"pKa OF IONIZABLE GROUPS\n";
      out.FILE2<<"  res    "
               <<"  pept "
               <<"  prot"<< std::endl;
      for(int jV0=0;jV0<oV0;jV0++){
         int iR0=str.V0[jV0].R0;
         double pKa_ref= str.V0[jV0].g/((2.302585)*physics_consts.ekT);
         out.FILE2<< std::setw( 4)<<(iR0+1)<<' '<<str.R0[iR0].aa
                  <<" ["<< std::setw( 5)<<pKa_ref<<']'
                  << std::setw( 6)<<str.V0[jV0].pKa<< std::endl;
      }
   }
//
//
// contribution to dG from locked ionizable groups
//
   aut.T[ 0].pK=( ( 7.50) -(.50)*dpK);
   aut.T[ 1].pK=( ( 4.00) +dpK);
   aut.T[ 2].pK=( ( 8.28) +dpK);
   aut.T[ 3].pK=( ( 4.40) +dpK);
   aut.T[ 4].pK=( ( 6.30) -dpK);
   aut.T[ 5].pK=( (10.50) -dpK);
   aut.T[ 6].pK=( ( 9.60) +dpK);
   aut.T[ 7].pK=( ( 3.80) +(.50)*dpK);
   aut.T[ 8].pK=( ( 6.30) -dpK);
   aut.T[ 0].sgn=( 1.00);
   aut.T[ 1].sgn=(-1.00);
   aut.T[ 2].sgn=( 1.00);
   aut.T[ 3].sgn=(-1.00);
   aut.T[ 4].sgn=(-1.00);
   aut.T[ 5].sgn=( 1.00);
   aut.T[ 6].sgn=( 1.00);
   aut.T[ 7].sgn=(-1.00);
   aut.T[ 8].sgn=(-1.00);
   aut.o_H6g.resize(nH6, (0.00));
   for(int iH6= 0;iH6<nH6;iH6++){
      double pH=aut.H6[iH6].pH;
      for(int iT= 0;iT<oT;iT++){
         double dg= aut.T[iT].sgn*(2.302585)*physics_consts.ekT
                   *( pH -aut.T[iT].pK);
         if( dg<=(0.00) )continue;
         int oL=aut.T[iT].nL;
         aut.H6g(iH6)+=(oL*dg);
      }
   }
//
//
// write titration diagnostic
//
   if( true ){
      out.FILE3<< std::fixed<< std::setprecision( 2);
      out.FILE3<<"PROFILE OF IONIZABLE GROUPS\n";
      out.FILE3<<"  res    "
               <<"    g  "
               <<"   pK  "
               <<" T"
               <<"   h  "
               <<"   w  "<<'\n';
      for(int jV0=0;jV0<oV0;jV0++){
         int iR0=str.V0[jV0].R0;
         out.FILE3<< std::setw( 4)<<(iR0+1)<<' '<<str.R0[iR0].aa
                  << std::setw( 7)<<str.V0[jV0].g
                  << std::setw( 7)<<(0.00)    //placeholder for pKa expt
                  << std::setw( 2)<<(str.V0[jV0].T+1)
                  << std::setw( 6)<<str.V0[jV0].hbc
                  << std::setw( 6)<<(1.00)<<'\n';
      }
      out.FILE3<<"FULLY PROTONATED\n";
      out.FILE3<<"    pp  "
               <<"    h   "
               <<"    ps  "
               <<"    ss  "<<'\n';
      out.FILE3<< std::setw( 9)<<(aut.I8[ 0].Fpp/energy_params.fac_pp)
               << std::setw( 9)<<(aut.I8[ 0].Fh /energy_params.fac_h )
               << std::setw( 9)<<(aut.I8[ 0].Fps/energy_params.fac_pp)
               << std::setw( 9)<<(aut.I8[ 0].Fss/energy_params.fac_pp)<<'\n';
      out.FILE3<<"TITRATION\n";
      for(int iH6=(nH6-1);iH6>-1;iH6--){
         out.FILE3<<" pH="<< std::setw( 6)<<aut.H6[iH6].pH
                  <<" dGmin="<< std::setw( 7)<<aut.H6[iH6].dGmin<<'\n';
         out.FILE3<<"        ";
         for(int jV0=0;jV0<oV0;jV0++){
            out.FILE3<<aut.V0a1(jV0);
         }
         out.FILE3<<"   dG  "
                  <<"   dF  "
                  <<"    g  "
                  <<"  p   "
                  <<"    e  "
                  <<"    pp  "
                  <<"    h   "
                  <<"    ps  "
                  <<"    ss  "<<'\n';
         int nK8=aut.H6[iH6].nK8;
         for(int iK8= 0;iK8<nK8;iK8++){
            out.FILE3<<" DEPROT=";
            for(int jV0=0;jV0<oV0;jV0++){
               out.FILE3<< std::setw( 1)<<aut.H6[iH6].K8[iK8].V0N2(jV0);
            }
            out.FILE3<< std::setw( 7)<<aut.H6[iH6].K8[iK8].dG
                     << std::setw( 7)<<aut.H6[iH6].K8[iK8].dF
                     << std::setw( 7)<<aut.H6[iH6].K8[iK8].g
                     << std::setprecision( 3)
                     << std::setw( 6)<<aut.H6[iH6].K8[iK8].p
                     << std::setprecision( 2)
                     << std::setw( 7)<<aut.H6[iH6].K8[iK8].e
                     << std::setw( 9)<<(aut.H6[iH6].K8[iK8].pp/
                                        energy_params.fac_pp)
                     << std::setw( 9)<<(aut.H6[iH6].K8[iK8].h /
                                        energy_params.fac_h )
                     << std::setw( 9)<<(aut.H6[iH6].K8[iK8].ps/
                                        energy_params.fac_pp)
                     << std::setw( 9)<<(aut.H6[iH6].K8[iK8].ss/
                                        energy_params.fac_pp)
                     <<'\n';
         }
      }
   }
   if( true ){
      out.FILE2<<"TITRATION\n";
      out.FILE2<<"  pH  "<<"  ";
      for(int jV0=0;jV0<oV0;jV0++){
         out.FILE2<<aut.V0a1(jV0);
      }
      out.FILE2<<"   dG  "
               <<"   dF  "
               <<"    g  "
               <<"  p   "
               <<"    pp  "
               <<"    h   "
               <<"    ps  "
               <<"    ss  "<<'\n';
      for(int iH6=(nH6-1);iH6>-1;iH6--){
         int jK8=aut.H6[iH6].jK8;
         out.FILE2<< std::setw( 6)<<aut.H6[iH6].pH<<"  ";
         for(int jV0=0;jV0<oV0;jV0++){
            out.FILE2<< std::setw( 1)<<aut.H6[iH6].K8[jK8].V0N2(jV0);
         }
         out.FILE2<< std::setw( 7)<<aut.H6[iH6].K8[jK8].dG
                  << std::setw( 7)<<aut.H6[iH6].K8[jK8].dF
                  << std::setw( 7)<<aut.H6[iH6].K8[jK8].g
                  << std::setprecision( 3)
                  << std::setw( 6)<<aut.H6[iH6].K8[jK8].p
                  << std::setprecision( 2)
                  << std::setw( 8)<<(a*aut.H6[iH6].K8[jK8].pp)
                  << std::setw( 8)<<(b*aut.H6[iH6].K8[jK8].h )
                  << std::setw( 8)<<(c*aut.H6[iH6].K8[jK8].ps)
                  << std::setw( 8)<<(d*aut.H6[iH6].K8[jK8].ss)
                  <<'\n';
      }
   }
// out.FILE1.close();
   out.FILE2.close();
   out.FILE3.close();

}
