#include "../phi/Coordinates.hh"
#include "../phi/Rotation_Matrix.hh"
#include <iostream>
#include <iomanip>
#include <cmath>

//
// construct
// P(alp,bet,gam)= rotation of bases by euler angles
// rotatedbasis(j)=P(i,j)initialbasis(i)
// rotatedcoords(i)=Ptranspose(i,j)initialcoords(j)
// p=Ptranspose
//
Rotation_Matrix::Rotation_Matrix(double alp,double bet,double gam){
   double Calp= std::cos( alp);
   double Salp= std::sin( alp);
   double Cbet= std::cos( bet);
   double Sbet= std::sin( bet);
   double Cgam= std::cos( gam);
   double Sgam= std::sin( gam);
   p[0][0]= Calp*Cbet*Cgam -Salp*Sgam;
   p[1][0]=-Calp*Cbet*Sgam -Salp*Cgam;
   p[2][0]= Calp*Sbet;
   p[0][1]= Salp*Cbet*Cgam +Calp*Sgam;
   p[1][1]=-Salp*Cbet*Sgam +Calp*Cgam;
   p[2][1]= Salp*Sbet;
   p[0][2]=-Sbet*Cgam;
   p[1][2]= Sbet*Sgam;
   p[2][2]= Cbet;
}
Rotation_Matrix::Rotation_Matrix(const Coordinates& a){
   double alp= a(0);
   double bet= a(1);
   double gam= a(2);
   double Calp= std::cos( alp);
   double Salp= std::sin( alp);
   double Cbet= std::cos( bet);
   double Sbet= std::sin( bet);
   double Cgam= std::cos( gam);
   double Sgam= std::sin( gam);
   p[0][0]= Calp*Cbet*Cgam -Salp*Sgam;
   p[1][0]=-Calp*Cbet*Sgam -Salp*Cgam;
   p[2][0]= Calp*Sbet;
   p[0][1]= Salp*Cbet*Cgam +Calp*Sgam;
   p[1][1]=-Salp*Cbet*Sgam +Calp*Cgam;
   p[2][1]= Salp*Sbet;
   p[0][2]=-Sbet*Cgam;
   p[1][2]= Sbet*Sgam;
   p[2][2]= Cbet;
}
//
//
// assign
//
Rotation_Matrix& Rotation_Matrix::operator=(const Rotation_Matrix& a){
   for(int i=0;i<3;i++){
      for(int j=0;j<3;j++){
         p[i][j]= a(i,j);
      }
   }
   return *this;
}
Rotation_Matrix& Rotation_Matrix::operator+=(const Rotation_Matrix& a){
   for(int i=0;i<3;i++){
      for(int j=0;j<3;j++){
         p[i][j]+=a(i,j);
      }
   }
   return *this;
}
Rotation_Matrix& Rotation_Matrix::operator-=(const Rotation_Matrix& a){
   for(int i=0;i<3;i++){
      for(int j=0;j<3;j++){
         p[i][j]-=a(i,j);
      }
   }
   return *this;
}
Rotation_Matrix& Rotation_Matrix::operator*=(const double a){
   for(int i=0;i<3;i++){
      for(int j=0;j<3;j++){
         p[i][j]*=a;
      }
   }
   return *this;
}
Rotation_Matrix& Rotation_Matrix::operator/=(const double a){
   for(int i=0;i<3;i++){
      for(int j=0;j<3;j++){
         p[i][j]/=a;
      }
   }
   return *this;
}
//
//
// initiate
//
void Rotation_Matrix::identity(){
   p[0][0]= (1.00);
   p[1][0]= (0.00);
   p[2][0]= (0.00);
   p[0][1]= (0.00);
   p[1][1]= (1.00);
   p[2][1]= (0.00);
   p[0][2]= (0.00);
   p[1][2]= (0.00);
   p[2][2]= (1.00);
   return;
}
void Rotation_Matrix::zero(){
   for(int i=0;i<3;i++){
      for(int j=0;j<3;j++){
         p[i][j]= (0.00);
      }
   }
   return;
}
//
//
// Choleski factor p=L*Ltranspose
//
bool Rotation_Matrix::lower(){
   if( p[0][0]<=(0.00) )return false;
   p[0][0]= std::sqrt( p[0][0]);
   for(int j=1;j<3;j++){
      p[j][0]= (p[0][j]/p[0][0]);
   }
   for(int i=1;i<3;i++){
      double z= (0.00);
      for(int k=0;k<i;k++){
         z+=p[i][k]*p[i][k];
      }
      if( (p[i][i]-z)<=(0.00) )return false;
      p[i][i]= std::sqrt( p[i][i] -z);
      for(int j=(i+1);j<3;j++){
         z= (0.00);
         for(int k=0;k<i;k++){
            z+=p[j][k]*p[i][k];
         }
         p[j][i]=( p[i][j] -z)/p[i][i];
      }
   }
   return true;
}
//
// extend generation matrix
// p= pr*tu*r
// p= p*r
//
void Rotation_Matrix::extend(Rotation_Matrix& pr,
                             const Rotation_Matrix& tu,
                             double chi){
   double B[3][3];
   for(int i=0;i<3;i++){
      for(int j=0;j<3;j++){
         B[i][j]= pr(i,0)*tu(0,j)
                 +pr(i,1)*tu(1,j)
                 +pr(i,2)*tu(2,j);
      }
   }
   double Cchi= std::cos( chi);
   double Schi= std::sin( chi);
   p[0][0]= B[0][0];
   p[1][0]= B[1][0];
   p[2][0]= B[2][0];
   p[0][1]= B[0][1]*Cchi +B[0][2]*Schi;
   p[1][1]= B[1][1]*Cchi +B[1][2]*Schi;
   p[2][1]= B[2][1]*Cchi +B[2][2]*Schi;
   p[0][2]=-B[0][1]*Schi +B[0][2]*Cchi;
   p[1][2]=-B[1][1]*Schi +B[1][2]*Cchi;
   p[2][2]=-B[2][1]*Schi +B[2][2]*Cchi;
   return;
}
void Rotation_Matrix::extend(double chi){
   double B[3][3];
   double Cchi= std::cos( chi);
   double Schi= std::sin( chi);
   B[0][1]= p[0][1]*Cchi +p[0][2]*Schi;
   B[1][1]= p[1][1]*Cchi +p[1][2]*Schi;
   B[2][1]= p[2][1]*Cchi +p[2][2]*Schi;
   B[0][2]=-p[0][1]*Schi +p[0][2]*Cchi;
   B[1][2]=-p[1][1]*Schi +p[1][2]*Cchi;
   B[2][2]=-p[2][1]*Schi +p[2][2]*Cchi;
   for(int i=0;i<3;i++){
      for(int j=1;j<3;j++){
         p[i][j]= B[i][j];
      }
   }
   return;
}
//
//
// euler angles
//
Coordinates Rotation_Matrix::euler(){
   double Cbet,Sbet,bet, Calp,Salp,alp, Cgam,Sgam,gam;
   Cbet= p[2][2];
   double zz=( (1.00) -Cbet*Cbet);
   if( zz<( 1.00e-25) )zz=( 1.00e-25);
   Sbet= std::sqrt( zz);
   bet= std::atan2(Sbet,Cbet);
   if( Sbet<(1.00e-12) ){
      gam= (0.00);
      if( Cbet>(0.00) ){
         Calp= p[0][0];
         Salp= p[0][1];
      }else{
         Calp=-p[0][0];
         Salp=-p[0][1];
      }
      alp= std::atan2(Salp,Calp);
   }else{
      Calp= (p[2][0]/Sbet);
      Salp= (p[2][1]/Sbet);
      alp= std::atan2(Salp,Calp);
      Cgam=-(p[0][2]/Sbet);
      Sgam= (p[1][2]/Sbet);
      gam= std::atan2(Sgam,Cgam);
   }
   Coordinates x;
   x(0)= alp;
   x(1)= bet;
   x(2)= gam;
   return x;
}
//
//
// global operators
//
Rotation_Matrix operator+(const Rotation_Matrix& a,
                          const Rotation_Matrix& b){
   Rotation_Matrix t=a;
   t+=b;
   return t;
}
Rotation_Matrix operator-(const Rotation_Matrix& a,
                          const Rotation_Matrix& b){
   Rotation_Matrix t=a;
   t-=b;
   return t;
}
Rotation_Matrix operator*(double a,
                          const Rotation_Matrix& b){
   Rotation_Matrix t=b;
   t*=a;
   return t;
}
Rotation_Matrix operator*(const Rotation_Matrix& a,
                          const Rotation_Matrix& b){
   Rotation_Matrix t;
   for(int i=0;i<3;i++){
      for(int j=0;j<3;j++){
         t(i,j)=( a(i,0)*b(0,j)
                 +a(i,1)*b(1,j)
                 +a(i,2)*b(2,j));
      }
   }
   return t;
}
std::ostream& operator<<(std::ostream& os,
                         const Rotation_Matrix& a){
   os<< std::fixed<< std::setprecision(3);
   for(int j=0;j<3;j++){
      for(int i=0;i<3;i++){
         os<< std::setw( 8)<<a(i,j);
      }
   }
   return os;
}
//
//
// global functions
//
Rotation_Matrix transpose(const Rotation_Matrix& a){
   Rotation_Matrix t;
   for(int i=0;i<3;i++){
      for(int j=0;j<3;j++){
         t(i,j)= a(j,i);
      }
   }
   return t;
}
Rotation_Matrix extend(const Rotation_Matrix& a,
                       double chi){
   Rotation_Matrix t;
   double Cchi= std::cos( chi);
   double Schi= std::sin( chi);
   for(int i=0;i<3;i++){
      t(i,0)= a(i,0);
      t(i,1)= a(i,1)*Cchi +a(i,2)*Schi;
      t(i,2)=-a(i,1)*Schi +a(i,2)*Cchi;
   }
   return t;
}
double dot(const Rotation_Matrix& a,
           const Rotation_Matrix& b){
   double z= (0.00);
   for(int i=0;i<3;i++){
      for(int j=0;j<3;j++){
         z+=a(i,j)*b(i,j);
      }
   }
   return z;
}
double eedot(const Rotation_Matrix& a,
             const Rotation_Matrix& b){
   Coordinates p;
   for(int j=0;j<3;j++){
      p(j)= (0.00);
      for(int i=0;i<3;i++){
         p(j)+=a(i,j)*b(i,j);
      }
   }
   double z= p(0);
   if( p(1)>(0.00) ){
      z+=( p(1) +p(2));
   }else{
      z-=( p(1) +p(2));
   }
// z/=(3.00);
   z=( z -(1.00))/(2.00);
   return z;
}
