#include "../dat/DAT_ARRAY_CONSTS.hh"
#include "../phi/Coordinates.hh"
#include "../phi/Multipoles.hh"
#include "../phi/Rotation_Matrix.hh"
#include "../phi/Rotation_Tensor.hh"
#include "../phi/Spherical_Harmonics.hh"
#include "../str/Output_Streams.hh"
#include <vector>
#include <iostream>
#include <iomanip>
#include <cmath>
//
//
// construct
//
Multipoles::Multipoles():
   oL(-1)
{
}
Multipoles::Multipoles(int o):
   oL(o),
   q_r((o+1)*(o+1)),
   q_i((o+1)*(o+1))
{
}
//
//
// assign
//
void Multipoles::operator=(const Multipoles& a){
   oL=a.order();
   if( oL<0 )return;
   int n=(oL+1)*(oL+1);
   if( n!=int(q_r.size()) ){
      q_r.resize(n);
      q_i.resize(n);
   }
   for(int j=0;j<n;j++){
      r(j)=a.r(j);
      i(j)=a.i(j);
   }
   return;
}
void Multipoles::truncate(int o,
                          const Multipoles& a){
   oL=o;
   int n=(o+1)*(o+1);
   if( n!=int(q_r.size()) ){
      q_r.resize(n);
      q_i.resize(n);
   }
   for(int j=0;j<n;j++){
      r(j)=a.r(j);
      i(j)=a.i(j);
   }
   return;
}
void Multipoles::operator*=(double a){
   int n=(oL+1)*(oL+1);
   for(int j=0;j<n;j++){
      q_r[j]*=a;
      q_i[j]*=a;
   }
   return;
}
void Multipoles::operator+=(const Multipoles& a){
   int o=( a.order()<=oL )? a.order(): oL;
   int n=(o+1)*(o+1);
   for(int j=0;j<n;j++){
      r(j)+=a.r(j);
      i(j)+=a.i(j);
   }
   return;
}
void Multipoles::operator-=(const Multipoles& a){
   int o=( a.order()<=oL )? a.order(): oL;
   int n=(o+1)*(o+1);
   for(int j=0;j<n;j++){
      r(j)-=a.r(j);
      i(j)-=a.i(j);
   }
   return;
}
void Surf_Match::operator=(const Surf_Match& a){
   for(int iX=0;iX<4;iX++){
      for(int iB=0;iB<6;iB++){
         for(int iL=0;iL<8;iL++){
            for(int iNN= 0;iNN<15;iNN++){
               XBNNLss(iX,iB,iNN,iL)= a.XBNNLss(iX,iB,iNN,iL);
            }
            for(int iN=0;iN<8;iN++){
               XBNLss(iX,iB,iN,iL)= a.XBNLss(iX,iB,iN,iL);
            }
            XBLss(iX,iB,iL)= a.XBLss(iX,iB,iL);
            XBLn(iX,iB,iL)= a.XBLn(iX,iB,iL);
            XBLd(iX,iB,iL)= a.XBLd(iX,iB,iL);
            XBLf(iX,iB,iL)= a.XBLf(iX,iB,iL);
            XBLg(iX,iB,iL)= a.XBLg(iX,iB,iL);
         }
         XBf(iX,iB)= a.XBf(iX,iB);
         XBg(iX,iB)= a.XBg(iX,iB);
      }
      Xf(iX)= a.Xf(iX);
      Xg(iX)= a.Xg(iX);
   }
   f= a.f;
   x= a.x;
   g= a.g;
   eolp= a.eolp;
   edepth= a.edepth;
   ecleft= a.ecleft;
   gfac= a.gfac;
   return;
}
//
//
// initiate
//
void Multipoles::zero(){
   int n=(oL+1)*(oL+1);
   for(int j=0;j<n;j++){
      q_r[j]= (0.00);
      q_i[j]= (0.00);
   }
   return;
}
//
//  q= rotation of a
//  q= rotation of q
//  q+= translation of a
//
void Multipoles::rotate(const DAT_ARRAY_CONSTS& array_consts,
                        const Rotation_Matrix& ROT,
                        const Multipoles& a){
   oL=a.order();
   if( oL<0 )return;
   int n=(oL+1)*(oL+1);
   if( n!=int(q_r.size()) ){
      q_r.resize(n);
      q_i.resize(n);
   }

   Rotation_Tensor d(array_consts,
                     oL,
                     ROT);

   for(int L=0;L<=oL;L++){
      for(int M1=-L,j1=(L*L);M1<=L;M1++,j1++){
         double zr= (0.00);
         double zi= (0.00);
         for(int M2=-L,j2=(L*L);M2<=L;M2++,j2++){
            zr+=( d.r(M1,M2,L)*a.r(j2)
                 -d.i(M1,M2,L)*a.i(j2));
            zi+=( d.r(M1,M2,L)*a.i(j2)
                 +d.i(M1,M2,L)*a.r(j2));
         }
         r(j1)= zr;
         i(j1)= zi;
      }
   }
   return;
}
void Multipoles::rotate(const DAT_ARRAY_CONSTS& array_consts,
                        const Rotation_Matrix& ROT){
   double t_r[64];
   double t_i[64];
   int n=(oL+1)*(oL+1);
   for(int j=0;j<n;j++){
      t_r[j]= q_r[j];
      t_i[j]= q_i[j];
   }

   Rotation_Tensor d(array_consts,
                     oL,
                     ROT);

   for(int L=0;L<=oL;L++){
      for(int M1=-L,j1=(L*L);M1<=L;M1++,j1++){
         double zr= (0.00);
         double zi= (0.00);
         for(int M2=-L,j2=(L*L);M2<=L;M2++,j2++){
            zr+=( d.r(M1,M2,L)*t_r[j2]
                 -d.i(M1,M2,L)*t_i[j2]);
            zi+=( d.r(M1,M2,L)*t_i[j2]
                 +d.i(M1,M2,L)*t_r[j2]);
         }
         r(j1)= zr;
         i(j1)= zi;
      }
   }
   return;
}
void Multipoles::translate(const DAT_ARRAY_CONSTS& array_consts,
                           const Coordinates& TRANS,
                           const Multipoles& a){
   double Lr[8];                //R**L

   double d= TRANS.r();
   Lr[ 0]= (1.00);
   Lr[ 1]= d;
   for(int L=2;L<=oL;L++){
      Lr[L  ]= d*Lr[L-1];
   }

   double CT= (TRANS(2)/d);
   double ST= std::sqrt( (1.00) -CT*CT);
   double CP,SP;
   if( ST>(1.00e-12) ){
      CP= (TRANS(0)/d)/ST;
      SP= (TRANS(1)/d)/ST;
   }else{
      CP= (1.00);
      SP= (0.00);
   }
   Spherical_Harmonics H;
   H.populate(array_consts,oL,CT,ST,CP,SP);

   for(int L=0;L<=oL;L++){
      int L1max=( L<=a.order() )? L: a.order();
      for(int L1=0;L1<=L1max;L1++){
         int L2=(L-L1);
         for(int M=-L,j=(L*L);M<=L;M++,j++){
            for(int M1=-L1,j1=(L1*L1);M1<=L1;M1++,j1++){
               int M2=(M-M1);
               if( M2> L2 )continue;
               if( M2<-L2 )continue;
               int j2=(L2*L2+L2+M2);
               double a_r= a.r(j1);
               double a_i= a.i(j1);
               double H_r= H.r(L2,M2);
               double H_i=-H.i(L2,M2);
               double z= Lr[L2]*array_consts.C[j1][j2];
               r(j)+=( a_r*H_r -a_i*H_i)*z;
               i(j)+=( a_r*H_i +a_i*H_r)*z;
            }
         }
      }
   }
   return;
}
//
//
// score match between 2 surfaces
//
void Surf_Match::fshpi(const DAT_ARRAY_CONSTS& array_consts,
                       Output_Streams& out,
                       const std::vector<Multipoles>& o_BXNs,
                       const std::vector<Multipoles>& o_BXs,
                       const std::vector<double>& o_Bu){
//
//
// components of distance metric
//
   for(int iX=0;iX<4;iX++){
      for(int iB=0;iB<6;iB++){
         int i=( iB*4 +iX);
         for(int NN= 0;NN<15;NN++){
            for(int L=0;L<8;L++){
               XBNNLss(iX,iB,NN, L)= (0.00);
            }
         }
         for(int a=0;a<8;a++){
            int j=( iB*32 +iX*8 +a);
            for(int b=a;b<8;b++){
               int k=( iB*32 +iX*8 +b);
               int NN=(a+b);
               double c=( b==a )? (1.00): (2.00);
               for(int L=0;L<8;L++){
                  double z= (0.00);
                  for(int M=-(L-b);M<=(L-b);M++){
                     z+=( o_BXNs[j].r(L, M)*o_BXNs[k].r(L,-M)
                         -o_BXNs[j].i(L, M)*o_BXNs[k].i(L,-M))
                        *array_consts.SGN( M);
                  }
                  XBNNLss(iX,iB,NN, L)+=(c*z);
               }
            }
         }
         for(int N=0;N<8;N++){
            int j=( iB*32 +iX*8 +N);
            for(int L=0;L<8;L++){
               double z= (0.00);
               for(int M=-(L-N);M<=(L-N);M++){
                  z+=( o_BXNs[j].r(L, M)*o_BXs[i].r(L,-M)
                      -o_BXNs[j].i(L, M)*o_BXs[i].i(L,-M))
                     *array_consts.SGN( M);
               }
               if( iX<2 ){
                  XBNLss(iX,iB, N, L)= (-2.00)*z;
               }else{
                  XBNLss(iX,iB, N, L)= ( 2.00)*z;
               }
            }
         }
         for(int L=0;L<8;L++){
            double z= (0.00);
            for(int M=-L;M<=L;M++){
               z+=( o_BXs[i].r(L, M)*o_BXs[i].r(L,-M)
                   -o_BXs[i].i(L, M)*o_BXs[i].i(L,-M))
                  *array_consts.SGN( M);
            }
            XBLss(iX,iB, L)= z;
         }
      }
   }
//
//
// weight as a func of BL
//
   double Xw[ 4];
   {
      double z= ( 2.50);
      Xw[ 0]= (1.00)/z;
      Xw[ 1]= ( .25)/z;
      Xw[ 2]= (1.00)/z;
      Xw[ 3]= ( .25)/z;
   }
   double Bw[ 6];
   {
      double z= ( 4.25);
      Bw[ 0]= ( .75)/z;
      Bw[ 1]= (1.00)/z;
      Bw[ 2]= (1.00)/z;
      Bw[ 3]= ( .75)/z;
      Bw[ 4]= ( .50)/z;
      Bw[ 5]= ( .25)/z;
   }
   double Lw[ 8];
   {
      double z= ( 8.00);
      Lw[ 0]= (1.00)/z;
      Lw[ 1]= (1.00)/z;
      Lw[ 2]= (1.00)/z;
      Lw[ 3]= (1.00)/z;
      Lw[ 4]= (1.00)/z;
      Lw[ 5]= (1.00)/z;
      Lw[ 6]= (1.00)/z;
      Lw[ 7]= (1.00)/z;
   }
   double XBLw[ 4][ 6][ 8];
   for(int iX=0;iX<4;iX++){
      for(int iB=0;iB<6;iB++){
         for(int L=0;L<8;L++){
            XBLw[iX][iB][ L]= (24.00)*Xw[iX]*Bw[iB]*Lw[ L];
         }
      }
   }
//
//
// f for zero displacement
//
   f= (0.00);
   for(int iX=0;iX<4;iX++){
      Xf(iX)= (0.00);
      for(int iB=0;iB<6;iB++){
         XBf(iX,iB)= (0.00);
         for(int L=0;L<8;L++){
            double den=( XBNNLss(iX,iB, 0, L) +XBLss(iX,iB, L));
            double num=( den +XBNLss(iX,iB, 0, L));
            XBLn(iX,iB, L)= num;
            XBLd(iX,iB, L)= den;
            if( den<( 1.00e-5) )den= ( 1.00e-5);
            XBLf(iX,iB, L)= XBLw[iX][iB][ L]*( (num/den) -( 1.00));
            XBf(iX,iB)+=XBLf(iX,iB, L);
         }
         Xf(iX)+=XBf(iX,iB);
      }
      f+=Xf(iX);
   }
//
//
// g for optimal displacement
//
   double BNz[ 6][15];
   for(int iB=0;iB<6;iB++){
      BNz[iB][ 0]= (1.00);
   }
   x= (  .00);
   g= f;
   for(int iX=0;iX<4;iX++){
      for(int iB=0;iB<6;iB++){
         for(int L=0;L<8;L++){
            XBLg(iX,iB, L)= XBLf(iX,iB, L);
         }
      }
   }
   double t( 0.00);
   double XBLt[ 4][ 6][ 8];
   for(double z=(-0.38);z<( 0.52);z+=(  .02)){
      if( std::abs( z)<( 1.00e-2) )continue;
      for(int iB=0;iB<6;iB++){
         double Bz= o_Bu[iB]*z;
         for(int N= 1;N<15;N++){
            BNz[iB][ N]=BNz[iB][N-1]*Bz;
         }
      }
      double t= (0.00);
      for(int iX=0;iX<4;iX++){
         for(int iB=0;iB<6;iB++){
            for(int L=0;L<8;L++){
               double den= XBLd(iX,iB, L);
               for(int N= 1;N<15;N++){
                  den+=XBNNLss(iX,iB, N, L)*BNz[iB][ N];
               }
               double num=( den +XBLn(iX,iB, L) -XBLd(iX,iB, L));
               for(int N=1;N<8;N++){
                  num+=XBNLss(iX,iB, N, L)*BNz[iB][ N];
               }
               if( den<( 1.00e-5) )den= ( 1.00e-5);
               XBLt[iX][iB][ L]= XBLw[iX][iB][ L]*( (num/den) -( 1.00));
               t+=XBLt[iX][iB][ L];
            }
         }
      }
      if( t< g ){
         x= z;
         g= t;
         for(int iX=0;iX<4;iX++){
            for(int iB=0;iB<6;iB++){
               for(int L=0;L<8;L++){
                  XBLg(iX,iB, L)= XBLt[iX][iB][ L];
               }
            }
         }
      }
   }
//
//
// s, h, p, i
//
   for(int iX=0;iX<4;iX++){
      Xg(iX)= (0.00);
      for(int iB=0;iB<6;iB++){
         XBg(iX,iB)= (0.00);
         for(int L=0;L<8;L++){
            XBg(iX,iB)+=XBLg(iX,iB, L);
         }
         Xg(iX)+=XBg(iX,iB);
      }
   }
//
//
// penalize overlap
//
   for(int iB=0;iB<6;iB++){
      double Bx= o_Bu[iB]*( x +( .15));
      for(int N= 1;N<15;N++){
         BNz[iB][ N]=BNz[iB][N-1]*Bx;
      }
   }
   {
//    out.FILE3<<std::fixed<<std::setprecision( 2);
      int iX=0;
      Multipoles a(7),d(7),b(7);
      double w= (1.00)/std::sqrt( (2.00));
      eolp= (0.00);
      double Beolp[ 6];
      for(int iB=0;iB<6;iB++){
//       out.FILE3<<" iB="<<std::setw( 2)<<iB<<'\n';
         int j=(iB*32);
         a.zero();
         for(int N=0;N<8;N++){
            d=o_BXNs[j+N];
            d*=BNz[iB][ N];
            a+=d;
         }
         double La[ 8];
         for(int L=0;L<8;L++){
            double z= (0.00);
            for(int M=-L;M<=L;M++){
               z+=( a.r(L, M)*a.r(L,-M)
                   -a.i(L, M)*a.i(L,-M))
                  *array_consts.SGN( M);
            }
            La[ L]= std::sqrt( z);
            if( La[ L]<( 1.00) )La[ L]= ( 1.00);
         }
         b=o_BXs[iB*4];
         double Lb[ 8];
         for(int L=0;L<8;L++){
            double z= (0.00);
            for(int M=-L;M<=L;M++){
               z+=( b.r(L, M)*b.r(L,-M)
                   -b.i(L, M)*b.i(L,-M))
                  *array_consts.SGN( M);
            }
            Lb[ L]= std::sqrt( z);
            if( Lb[ L]<( 1.00) )Lb[ L]= ( 1.00);
         }
         double a_c[8][8],a_s[8][8];
         double b_c[8][8],b_s[8][8];
         for(int L=2;L<8;L++){
//          out.FILE3<<" L="<<std::setw( 1)<<L;
            for(int M=1;M<=L;M++){
               if( (L+M)%2==0 )continue;
               a_c[L][M]= w*( a.r(L, M)*array_consts.SGN(M) +a.r(L,-M));
               a_s[L][M]= w*( a.i(L, M)*array_consts.SGN(M) -a.i(L,-M));
//             out.FILE3<<"[M="<<std::setw( 1)<<M<<']'
//                      <<" a_c="<<std::setw( 8)<<a_c[L][M]
//                      <<" a_s="<<std::setw( 8)<<a_s[L][M];
            }
//          out.FILE3<<'\n';
//          out.FILE3<<"    ";
            for(int M=1;M<=L;M++){
               if( (L+M)%2==0 )continue;
               b_c[L][M]= w*( b.r(L, M)*array_consts.SGN(M) +b.r(L,-M));
               b_s[L][M]= w*( b.i(L, M)*array_consts.SGN(M) -b.i(L,-M));
//             out.FILE3<<"     "
//                      <<" b_c="<<std::setw( 8)<<b_c[L][M]
//                      <<" b_s="<<std::setw( 8)<<b_s[L][M];
            }
//          out.FILE3<<'\n';
         }
         Beolp[iB]= (0.00);
         for(int L=1;L<8;L+=2){
            double dr=( (b.r(L, 0)/Lb[ L])
                       -(a.r(L, 0)/La[ L]));
            dr/=((  .50)*double( L));
            if( dr<( 0.00) ){
               Beolp[iB]-=dr;
            }
//          out.FILE3<<" (LM)=("<<std::setw( 1)<<L
//                              <<std::setw( 1)<<0<<')'
//                   <<" a_r="<<std::setw( 8)<<(a.r(L, 0)/La[ L])
//                   <<" b_r="<<std::setw( 8)<<(b.r(L, 0)/Lb[ L])
//                   <<" dr="<<std::setw( 8)<<dr
//                   <<"  Beolp[iB]="<<std::setw( 8)<<Beolp[iB]<<'\n';
         }
         for(int L=2;L<8;L++){
            for(int M=1;M<=L;M++){
               if( (L+M)%2==0 )continue;
               double sgnc= b.r( 1, 0)*b_c[L][M];
               if      ( sgnc<( -.05) ){
                  sgnc= (-1.00);
               }else if( sgnc>(  .05) ){
                  sgnc= ( 1.00);
               }else{
                  sgnc= ( 0.00);
               }
               double dc=sgnc*( (b_c[L][M]/Lb[ L])
                               -(a_c[L][M]/La[ L]));
               dc/=((  .50)*double( L+M));
               if( dc<( 0.00) ){
                  Beolp[iB]-=dc;
               }
//             out.FILE3<<" (LM)=("<<std::setw( 1)<<L
//                                 <<std::setw( 1)<<M<<')'
//                      <<" a_c="<<std::setw( 8)<<(a_c[L][M]/La[ L])
//                      <<" b_c="<<std::setw( 8)<<(b_c[L][M]/Lb[ L])
//                      <<" sgnc="<<std::setw( 5)<<sgnc
//                      <<" dc="<<std::setw( 8)<<dc
//                      <<"  Beolp="<<std::setw( 8)<<Beolp[iB]<<'\n';
               double sgns= b.r( 1, 0)*b_s[L][M];
               if      ( sgns<( -.05) ){
                  sgns= (-1.00);
               }else if( sgns>(  .05) ){
                  sgns= ( 1.00);
               }else{
                  sgns= ( 0.00);
               }
               double ds=sgns*( (b_s[L][M]/Lb[ L])
                               -(a_s[L][M]/La[ L]));
               ds/=((  .50)*double( L+M));
               if( ds<( 0.00) ){
                  Beolp[iB]-=ds;
               }
//             out.FILE3<<" (LM)=("<<std::setw( 1)<<L
//                                 <<std::setw( 1)<<M<<')'
//                      <<" a_s="<<std::setw( 8)<<(a_s[L][M]/La[ L])
//                      <<" b_s="<<std::setw( 8)<<(b_s[L][M]/Lb[ L])
//                      <<" sgns="<<std::setw( 5)<<sgns
//                      <<" ds="<<std::setw( 8)<<ds
//                      <<"  Beolp="<<std::setw( 8)<<Beolp[iB]<<'\n';
            }
         }
         eolp+=Bw[iB]*Beolp[iB];
//       out.FILE3<<" iB="<<std::setw( 2)<<iB
//                <<" Beolp="<<std::setw( 8)<<Beolp[iB]
//                <<"  eolp="<<std::setw( 8)<<eolp<<'\n';
      }
   }
   return;
}
//
//
// global operators
//
Multipoles operator+(const Multipoles& a,
                     const Multipoles& b){
   Multipoles t=a;
   t+=b;
   return t;
}
Multipoles operator-(const Multipoles& a,
                     const Multipoles& b){
   Multipoles t=a;
   t-=b;
   return t;
}
double dot(const DAT_ARRAY_CONSTS& array_consts,
           const Multipoles& a,
           const Multipoles& b){
   int oL=a.order();
   double z= (0.00);
   for(int L=0;L<=oL;L++){
      for(int M=-L;M<=L;M++){
         z+=( a.r(L, M)*b.r(L,-M)
             -a.i(L, M)*b.i(L,-M))
            *array_consts.SGN( M);
      }
   }
   return z;
}
std::ostream& operator<<(std::ostream& os,
                         const Multipoles& a){
   int o=a.order();
   int n=(o+1)*(o+1);
   os<< std::fixed<< std::setprecision(2);
   os<<"[ L M r i]"<<'\n';
   for(int L=0;L<=o;L++){
      for(int M=-L,j=(L*L);M<=L;M++,j++){
         os<<'['<< std::setw( 2)<<L<< std::setw( 2)<<M
           << std::setw( 7)<<a.r(j)
           << std::setw( 7)<<a.i(j)<<']';
         if( ((j+1)%4==0)||(j==(n-1)) ){
            os<<'\n';
         }
      }
   }
   return os;
}
std::ostream& operator<<(std::ostream& os,
                         const Surf_Match& a){
   os<< std::fixed<< std::setprecision(2);
   os<<"    x  "
       "    f  "
       "    g  "
       "  eolp "
       "  edep "
       "  ecle "
     <<'\n';
   os<<std::setw( 7)<<a.x
     <<std::setw( 7)<<a.f
     <<std::setw( 7)<<a.g
     <<std::setw( 7)<<a.eolp
     <<std::setw( 7)<<a.edepth
     <<std::setw( 7)<<a.ecleft
     <<'\n';
   os<<" X"
       "   Xf  ["
       "    0  "
       "    1  "
       "    2  "
       "    3  "
       "    4  "
       "    5  "
     <<"]\n";
   for(int iX=0;iX<4;iX++){
      os<<std::setw( 2)<<iX
        <<std::setw( 7)<<a.Xf(iX)<<'[';
      for(int iB=0;iB<6;iB++){
         os<<std::setw( 7)<<a.XBf(iX,iB);
      }
      os<<"]\n";
   }
   os<<" X"
       "   Xg  ["
       "    0  "
       "    1  "
       "    2  "
       "    3  "
       "    4  "
       "    5  "
     <<"]\n";
   for(int iX=0;iX<4;iX++){
      os<<std::setw( 2)<<iX
        <<std::setw( 7)<<a.Xg(iX)<<'[';
      for(int iB=0;iB<6;iB++){
         os<<std::setw( 7)<<a.XBg(iX,iB);
      }
      os<<"]\n";
   }
   for(int iX=0;iX<4;iX++){
      for(int iB=0;iB<6;iB++){
         os<<" X B"
             "   XBf "
             "   XBg "
           <<"]\n";
         os<<std::setw( 2)<<iX
           <<std::setw( 2)<<iB
           <<std::setw( 7)<<a.XBf(iX,iB)
           <<std::setw( 7)<<a.XBg(iX,iB)
           <<'\n';
         os<<" XBNNLss=";
         for(int iL=0;iL<8;iL++){
            os<<std::setw( 7)<<a.XBNNLss(iX,iB, 0,iL);
         }
         os<<'\n';
         os<<"  XBNLss=";
         for(int iL=0;iL<8;iL++){
            os<<std::setw( 7)<<a.XBNLss(iX,iB, 0,iL);
         }
         os<<'\n';
         os<<"   XBLss=";
         for(int iL=0;iL<8;iL++){
            os<<std::setw( 7)<<a.XBLss(iX,iB,iL);
         }
         os<<'\n';
         os<<" zero displacement\n";
         os<<"    XBLn=";
         for(int iL=0;iL<8;iL++){
            os<<std::setw( 7)<<a.XBLn(iX,iB,iL);
         }
         os<<'\n';
         os<<"    XBLd=";
         for(int iL=0;iL<8;iL++){
            os<<std::setw( 7)<<a.XBLd(iX,iB,iL);
         }
         os<<'\n';
         os<<"    XBLf=";
         for(int iL=0;iL<8;iL++){
            os<<std::setw( 7)<<a.XBLf(iX,iB,iL);
         }
         os<<'\n';
         os<<" optimal displacement\n";
         os<<"    XBLg=";
         for(int iL=0;iL<8;iL++){
            os<<std::setw( 7)<<a.XBLg(iX,iB,iL);
         }
         os<<'\n';
      }
   }
   return os;
}
