#ifndef DEF_ENERGY_SURFACE
#define DEF_ENERGY_SURFACE

#include "../con/Subset_Contracted_System.hh"
#include "../dat/DAT_ARRAY_CONSTS.hh"
#include "../dat/DAT_ENERGY_PARAMS.hh"
#include "../dat/DAT_PHYSICS_CONSTS.hh"
#include "../dat/DAT_REGION_MAPS.hh"
#include "../dat/DAT_RESIDUE_MAPPINGS.hh"
#include "../fil/Search_Subspace.hh"
#include "../fil/Structure.hh"
#include "../med/Dielec_Continu.hh"
#include "../phi/Block_Pair.hh"
#include "../phi/Conf_Dependent_System.hh"
#include "../phi/Coordinates.hh"
#include "../phi/Multipoles.hh"
#include "../phi/Phi1_Automatic.hh"
#include "../phi/Phi_Automatic.hh"
#include "../phi/Rotation_Matrix.hh"
#include "../reg/Regularization_Target.hh"
#include "../set/Mechanical_System.hh"
#include "../str/Output_Streams.hh"
#include "../str/Thread_Options.hh"
#include <vector>
//
// energy surface (2nd order approximation to true energy surface) and conf
// base class
//
class Energy_Surface { /*U2=set of degrees of freedom*/
protected:
   int oU2;                             //maximum number of degrees of freedom
   std::vector<double> o_U2chi;         //generalized coordinates
   std::vector<double> o_U2g;           //1st derivative
   std::vector<double> o_U2U2a;         //2nd derivative

public:
   int nU2;                             //number
   double TOT;                          //function

   double& U2chi(int i){
      return o_U2chi.at( i);
   }
   const double& U2chi(int i) const {
      return o_U2chi.at( i);
   }
   double& U2g(int i){
      return o_U2g.at( i);
   }
   const double& U2g(int i) const {
      return o_U2g.at( i);
   }
   double& U2U2a(int i,int j){
      return o_U2U2a.at( i*oU2 +j);
   }
   const double& U2U2a(int i,int j) const {
      return o_U2U2a.at( i*oU2 +j);
   }
   double& U2U2a_up(int i,int j){
      if( i>j ){
         return o_U2U2a.at( j*oU2 +i);
      }else{
         return o_U2U2a.at( i*oU2 +j);
      }
   }

   Energy_Surface(int o):
      oU2(o),
      o_U2chi(o),
      o_U2g(o),
      o_U2U2a(o*o)
   {
   }
};
//
//
// energy surface and conf for regularization
//
class Energy_Surface5: public Energy_Surface {
public:
   /*energy components associated with deviation from target values*/
   double Fa;                   //atom coordinates
   double Fb;                   //bond lengths
   double Fc;                   //bond angles
   double Fd;                   //torsion angles

   void PHI(const DAT_PHYSICS_CONSTS& physics_consts,
            Output_Streams& out,
            const Regularization_Target& reg);
   Energy_Surface5(int o):
      Energy_Surface(o)
   {
   }
};
//
//
// energy surface and conf for func and derivative evaluation of restchm
//
class Energy_Surface2: public Energy_Surface {
public:
   class tZ0 { /*chains*/
   public:
      int U2a;                  //start index for trans+rot degs of free
      tZ0(){}
   };
   class tR0 { /*residues*/
   public:
      tR0(){}
   };
   class tQ2 { /*torsions [forward order]*/
   public:
      Rotation_Matrix c;        //unit vector along rotatable bond
      Rotation_Matrix backup_c; //
      Coordinates x;            //coords of base atom
      Coordinates backup_x;     //
      bool cut;                 //binary decision ignore forward group
      int U2;                   //map into degs of free
      bool pass;                //subset of torsions
      tQ2():pass(false){}
   };
   class tF2 { /*atom [backward order]*/
   public:
      double fq;                //factor of chg decrease by dielec medium
      int D;                    //index into condensed set of hb donors
      int A;                    //index into condensed set of hb acceptors
      tF2(){}
   };
   class tE8 { /*atom pair with torsion dependence*/
   public:
      int O8a;                  //start index
      int cO8;                  //number of intervening degs of free
      double dv;                //derivative of gaussian vol wrt rr
      tE8(){}
   };
   class tO8 { /*torsion of intervening strings for atom pairs E8*/
   public:
      int U2;                   //index into set of degrees of freedom
      double s;                 //dRR/dchi
      tO8(){}
   };
   class tR0R0 { /*peptide-peptide H-bond, H-->O*/
   public:
      double e;                 //interaction energy
      bool a;                   //R below 2.4 ang
      bool b;                   //R below threshold
      bool o;                   //R below overlap
      double C;                 //chains of peptide plane alignment
      double B;                 //patterns of H-bonds in helices or sheets
      double D;                 //chg mod by adjacent peptide planes
      tR0R0(){}
   };
   class tF2F2 { /*other H-bond, H-->O*/
   public:
      int iF2;                  //1st index of atom pair
      int jF2;                  //2nd index
      double e;                 //interaction energy
      bool a;                   //R below 2.4 ang
      bool b;                   //R below threshold
      bool o;                   //R below overlap
      double C;                 //chains of peptide plane alignment
      double B;                 //patterns of H-bonds in helices or sheets
      double D;                 //chg mod by adjacent peptide planes
      tF2F2(){}
   };
   class tE4 { /*pairs of H-bonds in extendable patterns*/
   public:
      int iF2;                  //donor 1st pair
      int jF2;                  //acceptor
      int kF2;                  //donor 2nd pair
      int lF2;                  //acceptor
      double f;                 //coeff
      tE4(){}
   };
   class tE9 { /*atom pair with torsion dependence*/
   public:
      int O9a;                  //start index
      int cO9;                  //number of intervening degs of free
      tE9(){}
   };
   class tO9 { /*torsion of intervening strings for atom pairs E9*/
   public:
      int U2;                   //index into set of degrees of freedom
      double s;                 //de/dchi
      tO9(){}
   };

protected:
   int oZ0;                                     //n of chains
   int oR0;                                     //n of residues
   int oQ2;                                     //n of variable torsions
   int oF2;                                     //n of interaction sites
   int oE1;                                     //
   std::vector<bool> o_U2sub;                   //constrain to target
   std::vector<double> o_U2ch0;                 //target chi
   std::vector<double> o_U2uca;                 //unit of circular arc
   std::vector<double> o_U2d;                   //displacement of coords
   std::vector<Block_Pair> o_Q2Q2e;             //MACRO=all E1, MICRO=E1part
   std::vector<Block_Pair> o_Q2Q2g;             //MACRO=!E1part
   std::vector<Block_Pair> o_backup_Q2Q2g;      //
   std::vector<Coordinates> o_Z0Q2y;            //Q2c[i]x(Q2x[i]-Z0x[j])
   std::vector<Coordinates> o_backup_Z0Q2y;     //
   std::vector<Rotation_Matrix> o_Z0Z0y;        //(d/d axis rot)(Z0x[i]-Z0x[j])
   double e1_Fr;                                //Fr for atom pair
   double e1_Fe;                                //
   double e1_Fs;                                //
   double e1_Fh;                                //
   Block_Pair e1_g;                             //Fr+Fe+Fs+Ft for atom pair
   Block_Pair e1_h;                             //
   std::vector<bool> o_E1pass;                  //f=contacting sub-block pairs
   std::vector<bool> o_E1part;                  //t=evaluate in microstep
   std::vector<bool> o_backup_E1part;           //subset of sub-block pairs
   std::vector<tR0R0> o_R0R0;                   //peptide-peptide H-bond, H-->O
   std::vector<int> o_F2F2E8;                   //map to string of torsions
   std::vector<int> o_F2F2E9;                   //map to string of torsions
   int oD;                                      //n of condensed donors
   int oA;                                      //n of condensed acceptors
   std::vector<int> o_DAF2F2;                   //map to non pept-pept H-bonds
   int oF2F2;                                   //n
   std::vector<tF2F2> o_F2F2;                   //non pept-pept H-bond, D-->A

   Block_Pair& Q2Q2e(int i,int j){
      return o_Q2Q2e.at(i*oQ2+j);
   }
   Block_Pair& Q2Q2g(int i,int j){
      return o_Q2Q2g.at(i*oQ2+j);
   }
   Block_Pair& backup_Q2Q2g(int i,int j){
      return o_backup_Q2Q2g.at(i*oQ2+j);
   }
   Coordinates& Z0Q2y(int i,int j){
      return o_Z0Q2y.at(i*oQ2+j);
   }
   Coordinates& backup_Z0Q2y(int i,int j){
      return o_backup_Z0Q2y.at(i*oQ2+j);
   }
   Rotation_Matrix& Z0Z0y(int i,int j){
      return o_Z0Z0y.at(i*oZ0+j-i*(i+1)/2);
   }
   bool backup_E1part(int i){
      return o_backup_E1part[ i];
   }
   void backup_E1part(int i,bool a){
      o_backup_E1part[ i]=a;
      return;
   }
   tR0R0& R0R0(int i,int j){
      return o_R0R0.at( i*oR0 +j);
   }
   int& F2F2E8(int i,int j){
      if( i<=j ){
         return o_F2F2E8[ i*oF2 -i*(i+1)/2 +j];
      }else{
         return o_F2F2E8[ j*oF2 -j*(j+1)/2 +i];
      }
   }
   int& F2F2E9(int i,int j){
      if( i<=j ){
         return o_F2F2E9[ i*oF2 -i*(i+1)/2 +j];
      }else{
         return o_F2F2E9[ j*oF2 -j*(j+1)/2 +i];
      }
   }
   int& DAF2F2(int i,int j){
      return o_DAF2F2.at( i*oA +j);
   }
   tF2F2& F2F2(int i){
      return o_F2F2.at( i);
   }

   void PHI_XQ(Phi_Automatic& aut,
               const DAT_ARRAY_CONSTS& array_consts,
               Output_Streams& out,
               const Mechanical_System& mol,
               const Subset_Contracted_System::tM3& con,
               const Conf_Dependent_System& dep);
   void PHI_WPAR(Phi_Automatic& aut,
                 const DAT_PHYSICS_CONSTS& physics_consts,
                 const Subset_Contracted_System::tM3& con);
   void PHI_PPAR(Phi_Automatic& aut,
                 const DAT_PHYSICS_CONSTS& physics_consts,
                 const DAT_ARRAY_CONSTS& array_consts,
                 const DAT_REGION_MAPS& region_maps,
                 const Mechanical_System& mol,
                 const Subset_Contracted_System::tM3& con,
                 const Conf_Dependent_System& dep);
   void PHI_RESTCH(Phi_Automatic& aut,
                   const DAT_PHYSICS_CONSTS& physics_consts,
                   const DAT_ARRAY_CONSTS& array_consts,
                   const DAT_ENERGY_PARAMS& energy_params,
                   const Thread_Options& opt,
                   Output_Streams& out,
                   const Subset_Contracted_System::tM3& con,
                   const Conf_Dependent_System& dep);
   void PHI_E1(Phi_Automatic& aut,
               const DAT_PHYSICS_CONSTS& physics_consts,
               const DAT_ARRAY_CONSTS& array_consts,
               const DAT_ENERGY_PARAMS& energy_params,
               const Subset_Contracted_System::tM3& con,
               int iZ0,
               int mQ2,
               int iQ2,
               int iE1);
   int PHI_L(double r,
             double d,
             int Lmax) const;
   void PHI_E(Phi_Automatic& aut,
              const DAT_ARRAY_CONSTS& array_consts,
              const Multipoles& a1,
              int kQ2,
              int jZ0,
              const Coordinates& x2,
              const Multipoles& a2,
              int iQ2,
              int iZ0,
              int LTE);
   void PHI_RESH(Phi_Automatic& aut,
                 const DAT_PHYSICS_CONSTS& physics_consts,
                 const DAT_ARRAY_CONSTS& array_consts,
                 const DAT_ENERGY_PARAMS& energy_params,
                 const Subset_Contracted_System::tM3& con,
                 int jF2,
                 int kQ2,
                 int jZ0,
                 int iE1,
                 int iF2,
                 int i_3,
                 int iQ2,
                 int iZ0,
                 bool EPASS);
   void PHI_ME(Phi_Automatic& aut,
               const DAT_ARRAY_CONSTS& array_consts,
               const Multipoles& a1,
               int LTE1,
               const Multipoles& a2,
               int LTE2,
               int LTE);
   void PHI_PART(Phi_Automatic& aut,
                 const DAT_ARRAY_CONSTS& array_consts,
                 int kQ2,
                 int jZ0,
                 const Coordinates& x2,
                 int iQ2,
                 int iZ0);
   void PHI_T1D(const Conf_Dependent_System& dep,
                int iQ2);
   void PHI_T2D(const Subset_Contracted_System::tM3& con,
                const Conf_Dependent_System& dep,
                int iZ0);
   void PHI_C(Phi_Automatic& aut,
              const DAT_ARRAY_CONSTS& array_consts,
              const DAT_ENERGY_PARAMS& energy_params,
              const Thread_Options& opt,
              const Subset_Contracted_System::tM3& con,
              int jF2,
              int jZ0,
              int iC1,
              int iF2,
              int iQ2,
              int iZ0);
   void PHI_PWM(Phi_Automatic& aut,
                const DAT_PHYSICS_CONSTS& physics_consts,
                const DAT_ARRAY_CONSTS& array_consts,
                const DAT_REGION_MAPS& region_maps,
                Output_Streams& out,
                const Mechanical_System& mol,
                const Subset_Contracted_System::tM3& con,
                const Conf_Dependent_System& dep);
   void PHI_E1_WM(Phi_Automatic& aut,
                  const DAT_PHYSICS_CONSTS& physics_consts,
                  const DAT_ARRAY_CONSTS& array_consts,
                  const Subset_Contracted_System::tM3& con,
                  int iZ0,
                  int mQ2,
                  int iQ2,
                  int iE1);
   void PHI_WM(Phi_Automatic& aut,
               const DAT_PHYSICS_CONSTS& physics_consts,
               const DAT_ARRAY_CONSTS& array_consts,
               const Subset_Contracted_System::tM3& con,
               int jF2,
               int kQ2,
               int jZ0,
               int iF2,
               int iQ2,
               int iZ0);
   void PHI_E8(Phi_Automatic& aut,
               const Subset_Contracted_System::tM3& con,
               int jF2,
               int kQ2,
               int jZ0,
               int iF2,
               int iQ2,
               int iZ0,
               int pE8);
   void PHI_O8(const Phi_Automatic& aut,
               const Subset_Contracted_System::tM3& con,
               int kQ2,
               int kZ0,
               int iF2,
               int iZ0,
               int pE8);
   void PHI_E1_P(Phi_Automatic& aut,
                 const DAT_PHYSICS_CONSTS& physics_consts,
                 const DAT_ARRAY_CONSTS& array_consts,
                 const Subset_Contracted_System::tM3& con,
                 int iZ0,
                 int mQ2,
                 int iQ2,
                 int iE1);
   void PHI_P(Phi_Automatic& aut,
              const DAT_PHYSICS_CONSTS& physics_consts,
              const DAT_ARRAY_CONSTS& array_consts,
              const Subset_Contracted_System::tM3& con,
              int jF2,
              int kQ2,
              int jZ0,
              int iF2,
              int iQ2,
              int iZ0);
   void PHI_PART1(Phi_Automatic& aut,
                  int kQ2,
                  int jZ0,
                  const Coordinates& x2,
                  int iQ2,
                  int iZ0);
   void PHI_E9(Phi_Automatic& aut,
               const Subset_Contracted_System::tM3& con,
               int jF2,
               int kQ2,
               int jZ0,
               int iF2,
               int iQ2,
               int iZ0,
               int pE9);
   void PHI_O9(const Phi_Automatic& aut,
               const Subset_Contracted_System::tM3& con,
               int kQ2,
               int kZ0,
               int iF2,
               int iZ0,
               int pE9);
   void PHI_CONT(Phi_Automatic& aut,
                 const Subset_Contracted_System::tM3& con);
   void PHI_G(const Phi_Automatic& aut,
              int iZ0,
              int iQ2);
   void PHI_A(const Phi_Automatic& aut,
              const Subset_Contracted_System::tM3& con,
              int jZ0,
              int jQ2,
              int iZ0,
              int iQ2);

   void PHIE_RESTCH(Phi_Automatic& aut,
                    const DAT_PHYSICS_CONSTS& physics_consts,
                    const DAT_ARRAY_CONSTS& array_consts,
                    const DAT_ENERGY_PARAMS& energy_params,
                    const Thread_Options& opt,
                    Output_Streams& out,
                    const Subset_Contracted_System::tM3& con,
                    const Conf_Dependent_System& dep);
   void PHIE_E1(Phi_Automatic& aut,
                const DAT_PHYSICS_CONSTS& physics_consts,
                const DAT_ARRAY_CONSTS& array_consts,
                const DAT_ENERGY_PARAMS& energy_params,
                const Subset_Contracted_System::tM3& con,
                int iZ0,
                int mQ2,
                int iQ2,
                int iE1);
   void PHIE_E(Phi_Automatic& aut,
               const DAT_ARRAY_CONSTS& array_consts,
               const Multipoles& a1,
               const Multipoles& a2,
               int LTE);
   void PHIE_RESH(Phi_Automatic& aut,
                  const DAT_PHYSICS_CONSTS& physics_consts,
                  const DAT_ARRAY_CONSTS& array_consts,
                  const DAT_ENERGY_PARAMS& energy_params,
                  const Subset_Contracted_System::tM3& con,
                  int jF2,
                  int jZ0,
                  int iE1,
                  int iF2,
                  int i_3,
                  int iQ2,
                  int iZ0,
                  bool EPASS);
   void PHIE_ME(Phi_Automatic& aut,
                const DAT_ARRAY_CONSTS& array_consts,
                const Multipoles& a1,
                int LTE1,
                const Multipoles& a2,
                int LTE2,
                int LTE);
   void PHIE_T1D(const Conf_Dependent_System& dep,
                 int iQ2);
   void PHIE_T2D(const Subset_Contracted_System::tM3& con,
                 const Conf_Dependent_System& dep,
                 int iZ0);
   void PHIE_C(Phi_Automatic& aut,
               const DAT_ENERGY_PARAMS& energy_params,
               const Thread_Options& opt,
               const Subset_Contracted_System::tM3& con,
               int jF2,
               int iC1,
               int iF2);
   void PHIE_PWM(Phi_Automatic& aut,
                 const DAT_PHYSICS_CONSTS& physics_consts,
                 Output_Streams& out,
                 const Subset_Contracted_System::tM3& con);
   void PHIE_GP(Phi_Automatic& aut,
                const DAT_PHYSICS_CONSTS& physics_consts,
                const DAT_REGION_MAPS& region_maps,
                Output_Streams& out,
                const Mechanical_System& mol,
                const Subset_Contracted_System::tM3& con,
                const Conf_Dependent_System& dep);

   void PHI3_XQ(Phi_Automatic& aut,
                const Subset_Contracted_System::tM3& con,
                const Conf_Dependent_System& dep);
   void PHI3_RESTCH(Phi_Automatic& aut,
                    const DAT_PHYSICS_CONSTS& physics_consts,
                    const DAT_ARRAY_CONSTS& array_consts,
                    Output_Streams& out,
                    const Mechanical_System& mol,
                    const Subset_Contracted_System::tM3& con);
   void PHI3_D(const Phi_Automatic& aut,
               const DAT_PHYSICS_CONSTS& physics_consts,
               const Subset_Contracted_System::tM3& con,
               int jZ0,
               int jQ2,
               int iZ0,
               int iQ2);
   void PHI3_G(int iZ0,
               int iQ2);
   void PHI3_A(const Phi_Automatic& aut,
               const DAT_ARRAY_CONSTS& array_consts,
               const Subset_Contracted_System::tM3& con,
               int jZ0,
               int jQ2,
               int iZ0,
               int iQ2);

public:
   std::vector<tZ0> Z0;         //start index for trans+rot degs of free
   std::vector<tR0> R0;         //residues
   std::vector<tQ2> Q2;         //torsions [forward order]
   std::vector<tF2> F2;         //atoms [reverse order]
   int iCYC;                    //index of cycle over subsets of degs of free
   /*control of execution*/
   bool TETHER;                 //constraint to idealized structure
   bool RECYCLE;                //f=macrostep, reuse some x and q
   bool QMEDIUM;                //full chg reduction approx to dielec medium
   bool MICRO;                  //step type {t=MICRO, f=MACRO}
   bool SURFACE;                //evaluate energy, 1st, and 2nd derivatives
   bool DIVERGENCE;             //divergence detected in microstep energy
   double ethresh;              //convergence threshold for multipole expansion
   int Lprof[8];                //histogram of maximum L in multipole expansions
   /*energy components*/
   double Fr;                   //repulsion+dispersion
   double Fe;                   //electrostatic
   double Fs;                   //disulfide bond
   double Ft;                   //intrinsic torsional
   double Fc;                   //distance constraint
   double Fb;                   //ring closure
   double Fh;                   //hydrophobic
   double Fw;                   //excluded volume approx to dielectric medium
   double Fm;                   //dielectric medium
   double Fps;                  //protein-solvent
   double Fss;                  //solvent-solvent
   double Fg;                   //entropic reduction, disallow regions
   double Fp;                   //polarization
   double Fp_a;                 // H-bonds
   double Fp_b;                 // multiple H-bonds to a donor or acceptor
   double Fp_c;                 // chains of peptide plane alignment
   double Fp_d;                 // modulation of peptide H-bonds by elem type
   double Fp_e;                 // nonpolar atoms in the electric field
   /*microstep expansion*/
   double macro_Fr;             //
   double macro_Fe;             //
   double macro_Fs;             //
   double macro_Fh;             //
   double backup_Fr;            //
   double backup_Fe;            //
   double backup_Fs;            //
   double backup_Fh;            //
   double betafrac;             //estimate of fractional change to step size
   /*gaussian Fm*/
   double fac_m2;               //coeff for Fm 2-bod
   double fac_m3;               //3-bod
   std::vector<tE8> E8;         //atom pairs with torsion dependence
   int nE8;                     //number
   std::vector<tO8> O8;         //strings of intervening torsions
   int nO8;                     //number
   /*Fp*/
   bool E4RESET;                //reset subsets b and E4
   std::vector<tE4> E4;         //pairs of H-bonds in extendable patterns
   int nE4;                     //number
   std::vector<tE9> E9;         //atom pairs with torsion dependence
   int nE9;                     //number
   std::vector<tO9> O9;         //strings of intervening torsions
   int nO9;                     //number

   bool U2sub(int i){
      return o_U2sub[ i];
   }
   void U2sub(int i,bool a){
      o_U2sub[ i]=a;
      return;
   }
   double& U2ch0(int i){
      return o_U2ch0.at( i);
   }
   double& U2uca(int i){
      return o_U2uca.at( i);
   }
   const double& U2uca(int i) const {
      return o_U2uca.at( i);
   }
   double& U2d(int i){
      return o_U2d.at( i);
   }
   const double& U2d(int i) const {
      return o_U2d.at( i);
   }
   bool E1pass(int i){
      return o_E1pass[ i];
   }
   void E1pass(int i,bool a){
      o_E1pass[ i]=a;
      return;
   }
   bool E1part(int i){
      return o_E1part[ i];
   }
   void E1part(int i,bool a){
      o_E1part[ i]=a;
      return;
   }
   const double& U2ch0(int i) const {
      return o_U2ch0.at( i);
   }

   void backup(){
      for(int iQ2= 0;iQ2<oQ2;iQ2++){
         for(int jQ2= 0;jQ2<oQ2;jQ2++){
            backup_Q2Q2g(iQ2,jQ2)=Q2Q2g(iQ2,jQ2);
         }
      }
      for(int iZ0= 0;iZ0<oZ0;iZ0++){
         for(int iQ2= 0;iQ2<oQ2;iQ2++){
            backup_Z0Q2y(iZ0,iQ2)=Z0Q2y(iZ0,iQ2);
         }
      }
      for(int iE1= 0;iE1<oE1;iE1++){
         backup_E1part(iE1,E1part(iE1));
      }
      for(int iQ2= 0;iQ2<oQ2;iQ2++){
         Q2[iQ2].backup_c=Q2[iQ2].c;
         Q2[iQ2].backup_x=Q2[iQ2].x;
      }
      backup_Fr= macro_Fr;
      backup_Fe= macro_Fe;
      backup_Fs= macro_Fs;
      backup_Fh= macro_Fh;
      return;
   }
   void restor(){
      for(int iQ2= 0;iQ2<oQ2;iQ2++){
         for(int jQ2= 0;jQ2<oQ2;jQ2++){
            Q2Q2g(iQ2,jQ2)=backup_Q2Q2g(iQ2,jQ2);
         }
      }
      for(int iZ0= 0;iZ0<oZ0;iZ0++){
         for(int iQ2= 0;iQ2<oQ2;iQ2++){
            Z0Q2y(iZ0,iQ2)=backup_Z0Q2y(iZ0,iQ2);
         }
      }
      for(int iE1= 0;iE1<oE1;iE1++){
         E1part(iE1,backup_E1part(iE1));
      }
      for(int iQ2= 0;iQ2<oQ2;iQ2++){
         Q2[iQ2].c=Q2[iQ2].backup_c;
         Q2[iQ2].x=Q2[iQ2].backup_x;
      }
      macro_Fr= backup_Fr;
      macro_Fe= backup_Fe;
      macro_Fs= backup_Fs;
      macro_Fh= backup_Fh;
      return;
   }

   void PHI2(Phi_Automatic& aut,
             const DAT_PHYSICS_CONSTS& physics_consts,
             const DAT_ARRAY_CONSTS& array_consts,
             const DAT_ENERGY_PARAMS& energy_params,
             const DAT_REGION_MAPS& region_maps,
             const Thread_Options& opt,
             Output_Streams& out,
             const Mechanical_System& mol,
             const Subset_Contracted_System::tM3& con,
             Conf_Dependent_System& dep);
   void PHI3(Phi_Automatic& aut,
             const DAT_PHYSICS_CONSTS& physics_consts,
             const DAT_ARRAY_CONSTS& array_consts,
             const DAT_ENERGY_PARAMS& energy_params,
             const Thread_Options& opt,
             Output_Streams& out,
             const Mechanical_System& mol,
             const Subset_Contracted_System::tM3& con,
             Conf_Dependent_System& dep);
   void PHIE(Phi_Automatic& aut,
             const DAT_PHYSICS_CONSTS& physics_consts,
             const DAT_ARRAY_CONSTS& array_consts,
             const DAT_ENERGY_PARAMS& energy_params,
             const DAT_REGION_MAPS& region_maps,
             const Thread_Options& opt,
             Output_Streams& out,
             const Mechanical_System& mol,
             const Subset_Contracted_System::tM3& con,
             Conf_Dependent_System& dep);
   void PHIG(Phi_Automatic& aut,
             const DAT_PHYSICS_CONSTS& physics_consts,
             const DAT_ARRAY_CONSTS& array_consts,
             const DAT_ENERGY_PARAMS& energy_params,
             const DAT_REGION_MAPS& region_maps,
             const Thread_Options& opt,
             Output_Streams& out,
             const Mechanical_System& mol,
             const Subset_Contracted_System::tM3& con,
             Conf_Dependent_System& dep);
   void MED(Phi_Automatic& aut,
            Dielec_Continu& med,
            const DAT_PHYSICS_CONSTS& physics_consts,
            const DAT_ARRAY_CONSTS& array_consts,
            const DAT_ENERGY_PARAMS& energy_params,
            const Thread_Options& opt,
            Output_Streams& out,
            const Structure& str,
            const Mechanical_System& mol,
            const Subset_Contracted_System::tM3& con,
            Conf_Dependent_System& dep);
   void QFAC(Phi_Automatic& aut,
             const DAT_PHYSICS_CONSTS& physics_consts,
             Output_Streams& out,
             const Mechanical_System& mol,
             const Subset_Contracted_System::tM3& con,
             Conf_Dependent_System& dep);

   Energy_Surface2(int z,int r,int q,int f,int e,int o):
      Energy_Surface(o),
      o_U2uca(o),
      o_U2d(o),
      oZ0(z),
      oR0(r),
      oQ2(q),
      oF2(f),
      oE1(e),
      o_U2sub(o,false),
      o_U2ch0(o),
      o_Q2Q2e(q*q),
      o_Q2Q2g(q*q),
      o_backup_Q2Q2g(q*q),
      o_Z0Q2y(z*q),
      o_backup_Z0Q2y(z*q),
      o_Z0Z0y(z*(z+1)/2),
      o_E1pass(e,false),
      o_E1part(e),
      o_backup_E1part(e),
      o_R0R0(r*r),
      Z0(z),
      R0(r),
      Q2(q),
      F2(f),
      TETHER(false),
      RECYCLE(false),
      fac_m2( .000),
      fac_m3( .375)
   {
   }
};
//
//
// macro and micro step minimization using r-sqrared expansion
//
class Energy_Surface4: public Energy_Surface2 {
public:
   class tC3 { /*atom pairs with distance constraint used in exact segment
                 deformation*/
   public:
      double r;                 //target distance (bohr)
      double a;                 //harmonic coeff (hartree/bohr**2)
      int N2F2[2];              //pair of atom indexes
      tC3(){}
      void swap(tC3& c){
         double z= r;
         r= c.r;
         c.r= z;
         z= a;
         a= c.a;
         c.a= z;
         for(int iN2=0;iN2<2;iN2++){
            int L=N2F2[iN2];
            N2F2[iN2]=c.N2F2[iN2];
            c.N2F2[iN2]=L;
         }
      }
   };
   class tE2 { /*atom pair within small distance of repulsion overlap*/
   public:
      double q;                 //r-squared/2
      double rho;               // F=( 1 +alp)/( (r/rho) +alp)
      double eps;               // G=(1.12)/( (r/rho)**7 +.12)
      double alp;               //Fr= eps(F**7)( G -2)
      int cO2;                  //number of intervening degrees of freedom
      tE2(){}
   };
   class tO2 { /*torsion of intervening strings for atom pairs E2*/
   public:
      int U2;                   //index into set of degrees of freedom
      double s;                 //dq/dchi_t=dot( ( x_b -x_a), dx_b/dchi_t)
      Coordinates t;            //dx_b/dchi_t=cross( c_t, ( x_b -x_BSEt))
      Coordinates u;            //dx_a/dchi_t=cross( c_t, ( x_a -x_BSEt))
      tO2(){}
   };

protected:
   std::vector<double> o_backup_U2chi;  //generalized coordinates
   std::vector<double> o_U2dchi;        //macro step displacement of coords
   /*screen energy*/
   int oB6;                             //number of base residues
   int oG6;                             //number of generated residues
   std::vector<bool> o_B6B6pass;        //base-base subset of residue pairs
   std::vector<bool> o_B6G6pass;        //base-generated
   std::vector<bool> o_G6G6pass;        //generated-generated

   bool B6B6pass(int i,int j) const {
      return o_B6B6pass[ i*oB6 +j -(i*(i+1))/2];
   }
   bool B6G6pass(int i,int j) const {
      return o_B6G6pass[ i*oG6 +j];
   }
   bool G6G6pass(int i,int j) const {
      return o_G6G6pass[ i*oG6 +j -(i*(i+1))/2];
   }

   void PHI1_R(Phi1_Automatic& aut,
               const DAT_PHYSICS_CONSTS& physics_consts,
               const DAT_ENERGY_PARAMS& energy_params,
               const Subset_Contracted_System::tM3& con,
               int jF2,int kQ2,int jZ0,int iE1,
               int iF2,int i_3,int iQ2,int iZ0);
   void PHI1_E2(Phi1_Automatic& aut,
                const Subset_Contracted_System::tM3& con,
                int kQ2,int jZ0,int iF2,int iQ2,int iZ0);
   void PHI1_O2(const Phi1_Automatic& aut,
                const Subset_Contracted_System::tM3& con,
                int kQ2,int kZ0,int iF2,int iZ0);
   void PHID_X(Phi_Automatic& aut,
               const Mechanical_System& mol,
               const Subset_Contracted_System::tM3& con,
               const Conf_Dependent_System& dep);
   void PHID_TC(Phi_Automatic& aut,
                const DAT_PHYSICS_CONSTS& physics_consts,
                const DAT_ARRAY_CONSTS& array_consts,
                const DAT_ENERGY_PARAMS& energy_params,
                const Thread_Options& opt,
                const Mechanical_System& mol,
                const Subset_Contracted_System::tM3& con,
                const Conf_Dependent_System& dep);
   void PHI0_R(Phi1_Automatic& aut,
               const Subset_Contracted_System::tM3& con,
               int jF2,int kQ2,int jZ0,
               int iF2,int iQ2,int iZ0);

public:
   int HBC;                             //include hydrogen bond constraints
                                        //(0=no,1=yes)
   double Ecut;                         //energy threshold for overlap
   int nC3;                             //number
   std::vector<tC3> C3;                 //distance constraints
   /*macro step r-squared expansion*/
   int nE2;                             //number
   std::vector<tE2> E2;                 //atom pairs within 1.125*RHO
   int nO2;                             //number
   std::vector<tO2> O2;                 //strings of intervening degs of free
   int nE3;                             //number of overlapping atom pairs
   /*macro step backup copy of last accepted step*/
   int backup_nE2;                      //number
   std::vector<tE2> backup_E2;          //atom pairs within 1.125*RHO
   int backup_nO2;                      //number
   std::vector<tO2> backup_O2;          //strings of intervening degs of free
   int backup_nE3;                      //number of overlapping atom pairs
   /*screen energy components*/
   double Fs1;                          //weighted restch energy
   double Fs2;                          //
   double Fs3;                          //residue-residue contact
   double Fs4;                          //2-bod electrostatic of charged groups
   double Fs5;                          //3-bod electrostatic of charged groups

   double& backup_U2chi(int i){
      return o_backup_U2chi.at( i);
   }
   const double& backup_U2chi(int i) const {
      return o_backup_U2chi.at( i);
   }
   double& U2dchi(int i){
      return o_U2dchi.at( i);
   }
   const double& U2dchi(int i) const {
      return o_U2dchi.at( i);
   }
   void setB6B6pass(int i,int j,bool a){
      o_B6B6pass[ i*oB6 +j -(i*(i+1))/2]=a;
      return;
   }
   void setB6G6pass(int i,int j,bool a){
      o_B6G6pass[ i*oG6 +j]=a;
      return;
   }
   void setG6G6pass(int i,int j,bool a){
      o_G6G6pass[ i*oG6 +j -(i*(i+1))/2]=a;
      return;
   }

   void PHI1_HBND(const DAT_PHYSICS_CONSTS& physics_consts,
                  const DAT_ENERGY_PARAMS& energy_params,
                  Output_Streams& out,
                  const Search_Subspace& sub,
                  const Mechanical_System& mol,
                  const Subset_Contracted_System::tM3& con,
                  Conf_Dependent_System& dep);
   void PHI1_REXP(Phi1_Automatic& aut,
                  const DAT_PHYSICS_CONSTS& physics_consts,
                  const DAT_ENERGY_PARAMS& energy_params,
                  const Mechanical_System& mol,
                  const Subset_Contracted_System::tM3& con,
                  Conf_Dependent_System& dep);
   void PHI1_DCHI(const DAT_PHYSICS_CONSTS& physics_consts,
                  const Mechanical_System& mol,
                  const Subset_Contracted_System::tM3& con);
   void PHI1(const DAT_PHYSICS_CONSTS& physics_consts,
             const Mechanical_System& mol,
             const Subset_Contracted_System::tM3& con);

   void PHID(Phi_Automatic& aut,
             const DAT_PHYSICS_CONSTS& physics_consts,
             const DAT_ARRAY_CONSTS& array_consts,
             const DAT_ENERGY_PARAMS& energy_params,
             const Thread_Options& opt,
             const Mechanical_System& mol,
             const Subset_Contracted_System::tM3& con,
             Conf_Dependent_System& dep);

   void PHI0_HBND(const DAT_PHYSICS_CONSTS& physics_consts,
                  Output_Streams& out,
                  const Search_Subspace& sub,
                  const Mechanical_System& mol,
                  const Subset_Contracted_System::tM3& con,
                  Conf_Dependent_System& dep,
                  int iCYC,
                  int shf1,int shf2);
   void PHI0_REXP(const DAT_PHYSICS_CONSTS& physics_consts,
                  const Search_Subspace& sub,
                  const Mechanical_System& mol,
                  const Subset_Contracted_System::tM3& con,
                  Conf_Dependent_System& dep,
                  int iCYC);
   void PHI0(const DAT_PHYSICS_CONSTS& physics_consts,
             const Mechanical_System& mol,
             const Subset_Contracted_System::tM3& con);

   void PHIS(const DAT_PHYSICS_CONSTS& physics_consts,
             const DAT_RESIDUE_MAPPINGS& residue_mappings,
             Output_Streams& out,
             const Mechanical_System& mol,
             const Subset_Contracted_System::tM3& con,
             const Conf_Dependent_System& dep);

   Energy_Surface4(int z,int r,int q,int f,int e,int b,int g,int o):
      Energy_Surface2(z,r,q,f,e,o),
      o_backup_U2chi(o),
      o_U2dchi(o),
      oB6(b),
      oG6(g),
      o_B6B6pass((b*(b+1))/2,false),
      o_B6G6pass(b*g,false),
      o_G6G6pass((g*(g+1))/2,false)
   {
   }
};

#endif
