#ifndef DEF_LOCAL_MINIMIZATION
#define DEF_LOCAL_MINIMIZATION

#include "../con/Subset_Contracted_System.hh"
#include "../dat/DAT_ARRAY_CONSTS.hh"
#include "../dat/DAT_ENERGY_PARAMS.hh"
#include "../dat/DAT_PHYSICS_CONSTS.hh"
#include "../dat/DAT_REGION_MAPS.hh"
#include "../dat/DAT_RESIDUE_MAPPINGS.hh"
#include "../fil/Search_Subspace.hh"
#include "../fil/Structure.hh"
#include "../mov/Lagrange_Multiplier.hh"
#include "../phi/Conf_Dependent_System.hh"
#include "../phi/Energy_Surface.hh"
#include "../phi/Phi1_Automatic.hh"
#include "../phi/Phi_Automatic.hh"
#include "../reg/Regularization_Target.hh"
#include "../set/Mechanical_System.hh"
#include "../str/Output_Streams.hh"
#include "../str/Thread_Options.hh"
#include <string>
#include <vector>

//
//
// sequence of microsteps and control base class
//
class Local_Minimization {
public:
   class tM2 { /*minimization step*/
   public:
      double tot;               //function value
      double z;                 //square of gradient
      double bet;               //target step length
      std::string type;         //step type, MICRO or MACRO
      double delF;              //predicted change in energy
      /*cumulative step*/
      double  d2;               //square of macro step length
      /*adjustment of Lagrange multiplier lamda*/
      double lam0;              //initial value
      double  s0;               //square of step length of lam0
      double ds0;               //derivative of S0 wrt lam
      double lam1[4];           //estimate of negative of lowest eigenvalue
      double  s1[4];            //square of step length of lam1
      double ds1[4];            //derivative of S1 wrt lam
      double lam2;              //value corresponding to target step length
      double  s2;               //square of step length of lam2
      double ds2;               //derivative of S2 wrt lam
      /*iterations in adjustment of lamda*/
      int k0;                   //energy evaluations in line search
      int k1;                   //lam0 st (A +lam0*I) is positive definite
      int k2;                   //decrease toward negative of lowest eigenvalue
      int k3;                   //increase to lam2
      int k4[4];                //selection of lowest eigenvalue
      int k5[4];                //lam1 st (A +lam1*I) is positive definite
      /*characterization of multipole expansion component*/
      double ethresh;           //(-delF*(1.0e-5))
      int Lprof[8];             //distribution of L specifying truncation
      tM2(){}
   };

protected:
   void initial_lamda(Lagrange_Multiplier& lag,
                      int ck1,
                      double beta1);

public:
   /*step length*/
   double BETA;                 //target
   double BMIN;                 //minimum target
   double BMAX;                 //maximum target
   double BDEL;                 //fraction controlling rate of adjustment
   /*convergence thresholds*/
   double EPS1;                 //for gradient
   double EPS2;                 //for square of step length
   /*minimization steps*/
   std::vector<tM2> M2;         //set
   int nM2;                     //maximum number
   int iM2;                     //index
   /*return state from minimization*/
   int ENDSTATE;                //error code for local minimization
   int iM2macro;                //most recent macrostep
   int cM2macro;                //target number of microsteps for macrostep

   Local_Minimization(int oM2):
      M2(oM2)
   {
   }
};
//
//
// regularization
//
class Local_Minimization5: public Local_Minimization {
private:
   std::vector<double> o_K1BETA;        //initial target step length

   void backup(Lagrange_Multiplier& lag,
               Energy_Surface5& ene);
   double target_beta();
   void update(const Lagrange_Multiplier& lag,
               Energy_Surface5& ene);

public:
   /*K1=set of subsets of degrees of freedom in regularization*/
   double& K1BETA(int i){
      return o_K1BETA.at( i);
   }

   int MOV(const DAT_PHYSICS_CONSTS& physics_consts,
           Output_Streams& out,
           Energy_Surface5& ene,
           const Regularization_Target& reg);
   void WRT(Output_Streams& out,
            const Energy_Surface5& ene,
            const Regularization_Target& reg);

   Local_Minimization5(int oM2,int oK1):
      Local_Minimization(oM2),
      o_K1BETA(oK1)
   {
   }
};
//
//
// minimization of restchm for a mechanical system
//
class Local_Minimization2: public Local_Minimization {
protected:
   std::vector<double> o_M3BETA;        //step size

   void backup(Lagrange_Multiplier& lag,
               Energy_Surface2& ene,
               bool PRE);
   double target_beta(bool MICRO);
   void update(const DAT_PHYSICS_CONSTS& physics_consts,
               const Mechanical_System& mol,
               const Subset_Contracted_System::tM3& con,
               const Lagrange_Multiplier& lag,
               Energy_Surface2& ene);

public:
   int iM3;                             //index into subsets of degs of free
   double& M3BETA(int i){
      return o_M3BETA.at( i);
   }

   int MOV(Phi_Automatic& ph2,
           const DAT_PHYSICS_CONSTS& physics_consts,
           const DAT_ARRAY_CONSTS& array_consts,
           const DAT_ENERGY_PARAMS& energy_params,
           const DAT_REGION_MAPS& region_maps,
           const Thread_Options& opt,
           Output_Streams& out,
           const Mechanical_System& mol,
           const Subset_Contracted_System::tM3& con,
           Conf_Dependent_System& dep,
           Energy_Surface2& ene);
   void WRT(const DAT_PHYSICS_CONSTS& physics_consts,
            const DAT_RESIDUE_MAPPINGS& residue_mappings,
            const Thread_Options& opt,
            Output_Streams& out,
            Structure& str,
            const Mechanical_System& mol,
            const Subset_Contracted_System::tM3& con,
            Conf_Dependent_System& dep,
            Energy_Surface2& ene);

   Local_Minimization2(int oM2,int oM3):
      Local_Minimization(oM2),
      o_M3BETA(oM3),
      iM3( 0)
   {
   }
};
//
//
// macrostep machinery used with r-squared expansion
//
class Local_Minimization4: public Local_Minimization2 {
public:
   class tM4 { /*minimization macro step*/
   public:
      double Ftot;              //function value
      /*energy components*/
      double Fr;                //repulsion +dispersion
      double Ft;                //intrinsic torsional
      double Fc;                //distance constraint
      double Fb;                //ring closure
      /*characterization of macro step*/
      double step;              //square of step length
      double dtot;              //predicted change in energy
      double frac;              //(Ftot[i+1]-Ftot[i])/dtot[i]
      int nE2;                  //number of atom pairs st R<(1.125)*RHO
      int nO2;                  //number of torsions in torsion paths
      int nE3;                  //number of overlaps
      int M2pt;                 //index of final microstep
      int end;                  //end state
      tM4(){}
   };

private:
   void o1backup(Energy_Surface4& ene);
   void o1accept(Energy_Surface4& ene);
   void o1reject(Energy_Surface4& ene);
   int MOV1_MOV(Phi_Automatic& ph2,
                const DAT_PHYSICS_CONSTS& physics_consts,
                const DAT_ARRAY_CONSTS& array_consts,
                const DAT_ENERGY_PARAMS& energy_params,
                const Thread_Options& opt,
                const Mechanical_System& mol,
                const Subset_Contracted_System::tM3& con,
                Conf_Dependent_System& dep,
                Energy_Surface4& ene,
                Lagrange_Multiplier& lag);
   void backup(Lagrange_Multiplier& lag,
               Energy_Surface4& ene,
               bool PRE);
   double target_beta(bool PRE);
   void initial_lamda(Lagrange_Multiplier& lag,
                      int ck1,
                      double beta1,
                      bool PRE);
   void update(const DAT_PHYSICS_CONSTS& physics_consts,
               const Mechanical_System& mol,
               const Subset_Contracted_System::tM3& con,
               const Lagrange_Multiplier& lag,
               Energy_Surface4& ene);
   void o0backup(Energy_Surface4& ene);
   void o0accept(Energy_Surface4& ene);
   void o0reject(Energy_Surface4& ene);
   int MOV0_MOV(const DAT_PHYSICS_CONSTS& physics_consts,
                const Mechanical_System& mol,
                const Subset_Contracted_System::tM3& con,
                Energy_Surface4& ene,
                Lagrange_Multiplier& lag);
   void MOV_OBETA();
   void MOV_OSTEP(const DAT_PHYSICS_CONSTS& physics_consts,
                  const Mechanical_System& mol,
                  const Subset_Contracted_System::tM3& con,
                  Energy_Surface4& ene);
   void MOV_ODELF(const Energy_Surface4& ene);

public:
   double BET1;                 //initial target step length
   int cM2_1;                   //initial max number of steps per macro step
   int cM4_1;                   //initial max number of macro steps
   double BET2;                 //initial target step length
   std::vector<tM4> M4;         //macro steps
   int iM4;                     //index of macro step
   int cM2;                     //max number of steps per macro step

   int MOV1(Phi1_Automatic& ph1,
            Phi_Automatic& ph2,
            const DAT_PHYSICS_CONSTS& physics_consts,
            const DAT_ARRAY_CONSTS& array_consts,
            const DAT_ENERGY_PARAMS& energy_params,
            const Thread_Options& opt,
            Output_Streams& out,
            const Search_Subspace& sub,
            const Mechanical_System& mol,
            const Subset_Contracted_System::tM3& con,
            Conf_Dependent_System& dep,
            Energy_Surface4& ene);
   void WRT1(const Thread_Options& opt,
             Output_Streams& out,
             const Mechanical_System& mol,
             const Subset_Contracted_System::tM3& con,
             Conf_Dependent_System& dep,
             Energy_Surface4& ene);
   int MOV0(const DAT_PHYSICS_CONSTS& physics_consts,
            Output_Streams& out,
            const Search_Subspace& sub,
            const Mechanical_System& mol,
            const Subset_Contracted_System::tM3& con,
            Conf_Dependent_System& dep,
            Energy_Surface4& ene,
            int iCYC,
            int shf1,int shf2);
   void WRT0(Output_Streams& out,
             const Energy_Surface4& ene);

   Local_Minimization4(int oM4,int oM2,int oM3):
      Local_Minimization2(oM2,oM3),
      M4(oM4)
   {
   }
};

#endif
