#ifndef DEF_LAGRANGE_MULTIPLIER
#define DEF_LAGRANGE_MULTIPLIER

#include "../phi/Energy_Surface.hh"
#include "../tra/Energy_Surf.hh"
#include <vector>

class Lagrange_Multiplier {
private:
   std::vector<double> o_U2chi;         //backup copy of generalized coordinates
   std::vector<double> o_U2U2a;         //backup copy of 2nd derivative
   std::vector<double> o_U2h;           //diagonal of 2nd derivative
   std::vector<double> o_U2y;           //construction+destruction outside inner
                                        // loop
   double& U2y(int i){
      return o_U2y.at( i);
   }

public:
   double lamda;                        //Lagrange multiplier
   int nU2;                             //number of degrees of freedom
   int aU2;                             //lowest bound of index range
   int bU2;                             //highest
   double TOT;                          //backup copy of function
   double hmean;                        //mean of eigenvalues of 2nd derivative
   double hsigma;                       //standard deviation
   double hbound;                       //lower bound
   std::vector<double> o_U2g;           //backup copy of 1st derivative
   std::vector<double> o_U2x;           //step as a function of lamda
   std::vector<double> o_U2dx;          //1st derivative of step wrt lamda
   int oQ;                              //size of inversion subspace
   std::vector<double> o_QU2n;          //eigenvec of A with lowest eigenval
   std::vector<double> o_Qgn;           //component of g along n
   std::vector<double> o_Qhn;           //component of A along n*n(transpose)
   std::vector<double> o_Qf;            //factor Qhn-->Qhn(-Qf), range=[-1, 1]
   std::vector<double> o_Qf0;           //components: continuity of subspace
   std::vector<double> o_Qf1;           // continuity of momentum
   std::vector<double> o_Qf2;           // decrease in e+m
   std::vector<bool> o_Qsub;            //invert energy surf within subspace
   int oQ_pre;                          //previous step
   std::vector<double> o_QU2n_pre;      //eigenvec of A with lowest eigenval
   std::vector<double> o_Qgn_pre;       //component of g along n
   std::vector<double> o_Qhn_pre;       //component of A along n*n(transpose)
   std::vector<double> o_U2g_pre;       //1st derivative
   std::vector<double> o_U2x_pre;       //step as a function of lamda
   int oP;                              //size of protected subspace
   std::vector<double> o_PU2n;          //basis vector of protected subspace
   std::vector<double> o_QPt;           //coeffs of P wrt Q basis
   std::vector<double> o_Pgn;           //component of g along n
   std::vector<double> o_Phn;           //component of A along n*n(transpose)

   double& U2chi(int i){
      return o_U2chi.at( i);
   }
   double& U2g(int i){
      return o_U2g.at( i);
   }
   double& U2U2a(int i,int j){
      return o_U2U2a.at( i*nU2 +j);
   }
   const double& U2U2a(int i,int j) const {
      return o_U2U2a.at( i*nU2 +j);
   }
   double& U2h(int i){
      return o_U2h.at( i);
   }
   double& U2x(int i){
      return o_U2x.at( i);
   }
   const double& U2x(int i) const {
      return o_U2x.at( i);
   }
   double& U2dx(int i){
      return o_U2dx.at( i);
   }
   double& QU2n(int i,int j){
      return o_QU2n.at( i*nU2 +j);
   }
   double& Qgn(int i){
      return o_Qgn.at( i);
   }
   double& Qhn(int i){
      return o_Qhn.at( i);
   }
   double& Qf(int i){
      return o_Qf.at( i);
   }
   double& Qf0(int i){
      return o_Qf0.at( i);
   }
   double& Qf1(int i){
      return o_Qf1.at( i);
   }
   double& Qf2(int i){
      return o_Qf2.at( i);
   }
   void Qsub(int i,bool a){
      o_Qsub[ i]=a;
   }
   bool Qsub(int i){
      return o_Qsub[ i];
   }
   double& QU2n_pre(int i,int j){
      return o_QU2n_pre.at( i*nU2 +j);
   }
   double& Qgn_pre(int i){
      return o_Qgn_pre.at( i);
   }
   double& Qhn_pre(int i){
      return o_Qhn_pre.at( i);
   }
   double& U2g_pre(int i){
      return o_U2g_pre.at( i);
   }
   double& U2x_pre(int i){
      return o_U2x_pre.at( i);
   }
   double& PU2n(int i,int j){
      return o_PU2n.at( i*nU2 +j);
   }
   double& QPt(int i,int j){
      return o_QPt.at( i*32 +j);
   }
   double& Pgn(int i){
      return o_Pgn.at( i);
   }
   double& Phn(int i){
      return o_Phn.at( i);
   }

   template<class T> void accept(const T& ene){
      for(int iU2= 0;iU2<nU2;iU2++){
         U2chi(iU2)= ene.U2chi(iU2);
      }
      TOT= ene.TOT;
      for(int iU2= 0;iU2<nU2;iU2++){
         U2g(iU2)= ene.U2g(iU2);
      }
      for(int iU2= 0;iU2<nU2;iU2++){
         for(int jU2=iU2;jU2<nU2;jU2++){
            U2U2a(iU2,jU2)= ene.U2U2a(iU2,jU2);
         }
      }
      for(int iU2= 0;iU2<nU2;iU2++){
         U2h(iU2)= ene.U2U2a(iU2,iU2);
      }
      return;
   }
   template<class T> void reject(T& ene){
      for(int iU2= 0;iU2<nU2;iU2++){
         ene.U2chi(iU2)= U2chi(iU2);
      }
      return;
   }

   void acceptp(const Energy_Surfp& enp){
      for(int iU2= 0;iU2<nU2;iU2++){
         U2chi(iU2)= enp.U3chi(iU2);
      }
      TOT= enp.TOT;
      for(int iU2= 0;iU2<nU2;iU2++){
         U2g(iU2)= enp.U3g(iU2);
      }
      for(int iU2= 0;iU2<nU2;iU2++){
         for(int jU2=iU2;jU2<nU2;jU2++){
            U2U2a(iU2,jU2)= enp.U3U3a(iU2,jU2);
         }
      }
      for(int iU2= 0;iU2<nU2;iU2++){
         U2h(iU2)= enp.U3U3a(iU2,iU2);
      }
      return;
   }
   void rejectp(Energy_Surfp& enp){
      for(int iU2= 0;iU2<nU2;iU2++){
         enp.U3chi(iU2)= U2chi(iU2);
      }
      return;
   }

   double gradient();
   void diagonal();
   bool lower();
   void step(std::vector<double>& U2p,
             double c,
             std::vector<double>& U2o);
   double length(double& dxx);
   double normalize();
   void lamdastep(double beta1,
                  double xx,
                  double dxx,
                  bool INFLATE=false);
   double lamdastep(double beta1,
                    bool INFLATE=false);
   double deltaf();

   Lagrange_Multiplier(int o):
      o_U2chi(o),
      o_U2U2a(o*o),
      o_U2h(o),
      o_U2y(o),
      nU2(o),
      aU2(  0),
      bU2(o-1),
      o_U2g(o),
      o_U2x(o,(0.00)),
      o_U2dx(o),
      o_QU2n(32*o),
      o_Qgn(32),
      o_Qhn(32),
      o_Qf(32),
      o_Qf0(32),
      o_Qf1(32),
      o_Qf2(32),
      o_Qsub(32),
      o_QU2n_pre(32*o),
      o_Qgn_pre(32),
      o_Qhn_pre(32),
      o_U2g_pre(o),
      o_U2x_pre(o),
      o_PU2n(32*o),
      o_QPt(32*32),
      o_Pgn(32),
      o_Phn(32)
   {
   }
};

#endif
