#ifndef DEF_IGOR_MODEL
#define DEF_IGOR_MODEL

#include "../dat/DAT_ARRAY_CONSTS.hh"
#include "../dat/DAT_IGOR_DATA.hh"
#include "../dat/DAT_PHYSICS_CONSTS.hh"
#include "../dat/DAT_RESIDUE_MAPPINGS.hh"
#include "../fil/Structure.hh"
#include "../phi/Coordinates.hh"
#include "../phi/Gaussian_Volume.hh"
#include "../phi/Rotation_Matrix.hh"
#include "../str/Output_Streams.hh"
#include "../str/Thread_Options.hh"
#include <string>
#include <vector>
#include <set>
#include <iostream>

class Igor_Model {
public:
//
//________________________________________________________________
   class tH8 { /*hash of 0,1,2, or 3-bod pattern*/
   public:
      class tH8J7 { /*block of impulses consistent with pattern*/
      public:
         int E7a;               //start index of block
         int cE7;               //number
         int B7a;               //start index of block
         int cB7;               //number
         tH8J7(){}
      };
   public:
      int oJ7;                  //number
      std::vector<tH8J7> J7;    //blocks of impulses
      tH8(){}
   };
//
//________________________________________________________________
   class tR0 { /*residue*/
   private:
      std::vector<double> o_C7p0;       //prob from res ising model
      std::vector<double> o_C7e0;       //energy
      std::vector<double> o_C7w;        //path constraint
      std::vector<double> o_C7bo;       //max (over K7) of score (eo+zo)
      std::vector<int> o_C7K7;          //index st bo is max
      std::vector<double> o_C7eo;       //e of patch crosslinks for max bo
      std::vector<double> o_C7zo;       //e of alt path in res ising model
      std::vector<double> o_C7po;       //p observed in K7 sample
      std::vector<double> o_C7go;       //update to 1-bod res impulses
   public:
      int L1;                   //residue type
      int O7;                   //cluster of residue types
      int P7;                   //cluster of residue types
      int zet;                  //sgn of charge
      int kap;                  //size of low dielectric volume
      Gaussian_Volume ef[4];    //elec field density sqrt(hart/bohr**3)
      Gaussian_Volume vf;       //volume density sc
      int lnk;                  //index of residue crosslinked in ss bond
      int C70;                  //most probable state
      int C7z;                  //path with highest score
      int C7def;                //scratch state
      Coordinates xca;          //pos of Calp for conf with highest score
      Coordinates xn;           //pos of N
      Coordinates xc;           //pos of C
      Coordinates nca;          //direction of chain at Calp
      Coordinates xsc;          //pos of sc interaction site
      double del;               //dist of b from a
      int C7;                   //sequence of ss states observed
      tR0():o_C7p0( 3),o_C7e0( 3),
            o_C7w( 3),
            o_C7bo( 3),o_C7K7( 3),o_C7eo( 3),o_C7zo( 3),
            o_C7po( 3),o_C7go( 3){}
      double& C7p0(int a){
         return o_C7p0.at(a-1);
      }
      double& C7e0(int a){
         return o_C7e0.at(a-1);
      }
      double& C7w(int a){
         return o_C7w.at(a-1);
      }
      double& C7bo(int a){
         return o_C7bo.at(a-1);
      }
      int& C7K7(int a){
         return o_C7K7.at(a-1);
      }
      double& C7eo(int a){
         return o_C7eo.at(a-1);
      }
      double& C7zo(int a){
         return o_C7zo.at(a-1);
      }
      double& C7po(int a){
         return o_C7po.at(a-1);
      }
      double& C7go(int a){
         return o_C7go.at(a-1);
      }
      const double& C7p0(int a) const {
         return o_C7p0.at(a-1);
      }
      const double& C7e0(int a) const {
         return o_C7e0.at(a-1);
      }
      void operator=(const tR0& a);
   };
//
//________________________________________________________________
   class tG7 { /*element*/
   private:
      std::vector<double> o_C7p0;       //probability from res ising model
      std::vector<double> o_C7e0;       //energy
   public:
      int mR0;                  //start res of elem
      int nR0;                  //end res of elem
      int T7;                   //elem type
      int V7;                   //elem type lower resolution
      int W7;                   //elem type lower resolution
      int C70;                  //index of most probable state
      tG7():o_C7p0( 3),o_C7e0( 3){}
      double& C7p0(int a){
         return o_C7p0.at(a-1);
      }
      double& C7e0(int a){
         return o_C7e0.at(a-1);
      }
      const double& C7p0(int a) const {
         return o_C7p0.at(a-1);
      }
      const double& C7e0(int a) const {
         return o_C7e0.at(a-1);
      }
      void operator=(const tG7& a);
   };
//
//________________________________________________________________
   class Composition { /*composition*/
   public:
      int oG7;                  //number
      std::vector<int> G7jR0;   //partition of residues into elements
      Composition(){}
      Composition(int g,
                  const std::vector<tG7>& a);
   };
//
//________________________________________________________________
   class tW4 { /*hydrophob patch*/
   public:
      class tW4J4 { /*res of hydrophob patch*/
      public:
         int R0;                //index of res
         char a1;               //1 char res name
         tW4J4(){}
      };
      class tW4K4 { /*elem in bod*/
      public:
         int G7;                //index of elem
         int X;                 //x position in sheet
         int Y;                 //y position in sheet
         int sgnx;              //direction of CO and NH along x
         int sgny;              //direction of strand along y
         tW4K4():X(-1),Y(-1),sgnx( 1),sgny( 1){}
      };
   public:
      int oJ4;                  //number
      std::vector<tW4J4> J4;    //residues of hydrophob patch
      int O1;                   //twist
      int hp;                   //sum over patch of L1doc
      int P3;                   //pattern
      int R0;                   //start index of enclosing res subset
      int L;                    //alt surf over J4
      int N;                    //order of patch with equiv twist
      int oK4;                  //number
      std::vector<tW4K4> K4;    //res strings composing elem
      int oX;                   //width of sheet
      int oY;                   //height of sheet
      double h;                 //measure of sheet polarity
      bool PAR;                 //contains adjacent parallel strands
      int pen;                  //penalty
      tW4():oX(-1),oY(-1),h( 0.00),PAR(false),pen( 0){}
      void operator=(const tW4& a);
   };
//
//________________________________________________________________
   class tK7 { /*high SCORE chain state*/
   public:
      class tK7G7 { /*element of partition*/
      public:
         int mR0;               //start res of elem
         int nR0;               //end res
         int T7;                //type
         int C7;                //ss state
         tK7G7(){}
         void operator=(const tG7& a);
      };
      class tK7R0 { /*residue*/
      public:
         Coordinates xca;       //pos of Calp for conf with highest score
         Coordinates xn;        //pos of N
         Coordinates xc;        //pos of C
         Coordinates xsc;       //pos of sc interaction site
         bool b;                //pack induced ss state stability
         tK7R0(){}
      };
   public:
      int oG7;                  //number
      std::vector<tK7G7> G7;    //elements of chain
      bool PACK;                //pack config exists
      int oR0;                  //number
      std::vector<tK7R0> R0;    //elements of chain
      double e[9];              //SCORE of partition
      double e3;                //T9[iT9].e[ 1]*5 (iT9=min st T9[].sub==true)
      bool b;                   //core stabilized by patch crosslinks
      int K7ord;                //map to pre sort order
      tK7(){}
      void operator=(const tK7& a);
   };
//
//________________________________________________________________
   class tM8 { /*subset of strands isolated from complement*/
   public:
      class tM8S8 { /*strand*/
      public:
         int G7;                    //index of elem
         tM8S8(){}
      };
   public:
      int oS8;                      //number
      std::vector<tM8S8> S8;        //strands in subset
      tM8(){}
      void operator=(const tM8& a);
  };
//
//________________________________________________________________
   class tY8;
   class tQ8 { /*backup of config search for sheet*/
   public:
      class tQ8D8 { /*high SCORE config*/
      public:
         class tQ8D8Z8 { /*strand of sheet*/
         public:
            char X;                     //position in sheet
            float e;                    //energy of alignment left and right
            char A;                     //alignment wrt following x
            char B;                     //alignment wrt previous x
            tQ8D8Z8(){}
         };
      public:
         std::vector<tQ8D8Z8> Z8;       //strands of sheet
         float e[3];                    //energy of config
         tQ8D8(){}
      };
   public:
      int oD8;                          //number of configs
      std::vector<tQ8D8> D8;            //configs
      int oZ8;                          //number of strands
      tQ8(){}
      void operator=(const tY8& a);
   };
//
//________________________________________________________________
   class tZ8 { /*strand*/
   public:
      class tZ8R0 { /*res of strand*/
      public:
         int X;                 //x position in sheet
         int Y;                 //y position in sheet
         int sgnx;              //direction of CO and NH along x
         int sgny;              //direction of strand along y
         tZ8R0(){}
      };
   public:
      int oR0;                  //number
      std::vector<tZ8R0> R0;    //res string of strand
      tZ8(){}
   };
//
//________________________________________________________________
   class tY8 { /*oU-strand subsystem*/
   public:
      class tY8D8 { /*high SCORE config*/
      public:
         class tY8D8Z8 { /*strand*/
         public:
            char X;                     //position in subsystem
            float e;                    //energy of alignment left and right
            char A;                     //alignment wrt right strand
            char B;                     //alignment wrt left strand
            tY8D8Z8(){}
         };
      public:
         std::vector<tY8D8Z8> Z8;       //strands
         double e[ 6];                  //energy of config
         tY8D8(){}
         void swap(tY8::tY8D8& a);
      };
   public:
      int oD8;                          //number
      std::vector<tY8D8> D8;            //set of configs
      int oZ8;                          //number of strands
      tY8(){}
      void operator=(const tY8& a);
      void operator=(const tQ8& a);
   };
//
//________________________________________________________________
   class tU8 { /*all subsystems containing oU strands*/
   public:
      class tU8Y8 { /*oU-strand subsystem*/
      public:
         class tU8Y8Z8 { /*strand of subsystem*/
         public:
            int S8;                     //index of strand
            int Y8;                     //index of (oU-1)-strand subsystem
            char X;                     //position in subsystem
            float e;                    //energy of alignment left and right
            char A;                     //alignment wrt right strand
            char B;                     //alignment wrt left strand
            tU8Y8Z8(){}
         };
      public:
         std::vector<tU8Y8Z8> Z8;       //strands of subsystem
         bool STABLE;                   //exists stable config
         bool FACTOR;                   //subsys +complement stable wrt union
         double e[3];                   //energy of config [F,F-Fp,Fp]
         tU8Y8(){}
         void operator=(const tY8::tY8D8& a);
      };
   public:
      int oY8;                          //number
      std::vector<tU8Y8> Y8;            //oU-strand subsystems
      tU8():oY8( 0){}
      void operator=(const tU8& a);
   };
//
//________________________________________________________________
   class tT8 { /*factor subconfigs*/
   public:
      int U8;                   //number of strands
      int Y8;                   //optimal config
      bool sub;                 //in optimal union of subconfigs
      double e[3];              //SCORE of subconfig [F,F-Fp,Fp]
      tT8(){}
      void swap(tT8& a);
      void operator=(const tT8& a);
   };
//
//________________________________________________________________
   class tV8 { /*sheet of optimal union of subconfigs*/
   public:
      class tV8X { /*x position*/
      public:
         std::string head0;         //header of x column
         std::string head1;         //header of x column
         std::string head2;         //header of x column
         tV8X():head0("  "),head1("  "),head2("  "){}
      };
      class tV8XY { /*xy position*/
      public:
         std::string alp;           //aa+sgnx
         tV8XY():alp("  "){}
      };
   public:
      int lX;                       //lower bound of x
      int uX;                       //upper bound of x
      int lY;                       //lower bound of y
      int uY;                       //upper bound of y
      std::vector<tV8X> o_X;        //x positions
      std::vector<tV8XY> o_XY;      //xy positions
      tV8X& X(int i){
         return o_X.at( i-lX);
      }
      tV8XY& XY(int i,int j){
         return o_XY.at( (i-lX)*(uY-lY+ 1) +(j-lY));
      }
      const tV8X& X(int i) const {
         return o_X.at( i-lX);
      }
      const tV8XY& XY(int i,int j) const {
         return o_XY.at( (i-lX)*(uY-lY+ 1) +(j-lY));
      }
      tV8(){}
      void operator=(const tV8& a);
   };
//
//________________________________________________________________
   class tM9 { /*subset of bods isolated from complement*/
   public:
      class tM9S9 { /*bod*/
      public:
         class tM9S9B4 { /*patch on bod*/
         public:
            int W4;                 //index of patch
            tM9S9B4(){}
         };
         class tM9S9K4 { /*elem in bod*/
         public:
            int G7;                 //index of elem
            int X;                  //x position in sheet
            int Y;                  //y position in sheet
            int sgnx;               //direction of CO and NH along x
            int sgny;               //direction of strand along y
            tM9S9K4(){}
         };
      public:
         int oB4;                   //number
         std::vector<tM9S9B4> B4;   //patches on bod
         int oK4;                   //number of res strings
         std::vector<tM9S9K4> K4;   //elems in bod
         tM9S9(){}
      };
   public:
      int oS9;                      //number
      std::vector<tM9S9> S9;        //bods in collection
      tM9(){}
      void operator=(const tM9& a);
  };
//
//________________________________________________________________
   class tY9;
   class tQ9 { /*backup of congig search for pack*/
   public:
      class tQ9D9 { /*high SCORE config*/
      public:
         class tQ9D9Z9 { /*bod*/
         public:
            class tQ9D9Z9K4 { /*elem in bod*/
            public:
               short G7;                //index of elem
               char X;                  //x position in sheet
               char Y;                  //y position in sheet
               char sgnx;               //direction of CO and NH along x
               char sgny;               //direction of strand along y
               Coordinates u;           //position of elem start
               Coordinates v;           //position of elem end
               Rotation_Matrix r;       //axes of elem in pack config
               tQ9D9Z9K4(){}
            };
         public:
            char X;                     //position in subsystem
            float e;                    //energy of alignment left and right
            short A;                    //alignment wrt left bod
            short B;                    //alignment wrt right bod
            Rotation_Matrix c;          //axes of bod in pack config
            Coordinates t;              //position of patch center
            std::vector<tQ9D9Z9K4> K4;  //elements in bod
            tQ9D9Z9(){}
         };
         class tQ9D9Q { /*local coord system within config*/
         public:
            Rotation_Matrix r;          //orientation of local axes
            Coordinates t;              //position of origin
            tQ9D9Q(){}
         };
      public:
         std::vector<tQ9D9Z9> Z9;       //bods
         std::vector<tQ9D9Q> Q;         //4 local coord systems
         double e[ 4];                  //energy of config
         tQ9D9():Q( 4){}
      };
   public:
      int oD9;                          //number
      std::vector<tQ9D9> D9;            //set of configs
      int oZ9;                          //number of bods
      tQ9(){}
      void operator=(const tY9& a);
   };
//
//________________________________________________________________
   class tZ9 { /*bod*/
   public:
      class tZ9K4 { /*elem*/
      public:
         short G7;              //index of elem
         char X;                //x pos in sheet
         char Y;                //y pos in sheet
         char sgnx;             //direction of CO and NH along x
         char sgny;             //direction of strand along y
         Coordinates u;         //pos of elem start in pack config
         Coordinates v;         //pos of elem end
         Rotation_Matrix r;     //local axes for elem in pack config
         tZ9K4(){}
      };
   public:
      int oK4;                  //number
      std::vector<tZ9K4> K4;    //elements in bod
      Rotation_Matrix c;        //local axes of bod in pack config
      Coordinates t;            //pos of local axes of bod
      tZ9(){}
   };
//
//________________________________________________________________
   class tY9 { /*oU-bod subsystem*/
   public:
      class tY9D9 { /*high SCORE config*/
      public:
         class tY9D9Z9 { /*bod*/
         public:
            class tY9D9Z9K4 { /*elem in bod*/
            public:
               short G7;                //index of elem
               char X;                  //x pos in sheet
               char Y;                  //y pos in sheet
               char sgnx;               //direction of CO and NH along x
               char sgny;               //direction of strand along y
               Coordinates u;           //pos of elem start in pack config
               Coordinates v;           //pos of elem end
               Rotation_Matrix r;       //local axes for elem in pack config
               tY9D9Z9K4(){}
            };
         public:
            char X;                     //pos in subsystem
            float e;                    //energy of alignment left and right
            short A;                    //alignment wrt left bod
            short B;                    //alignment wrt right bod
            Rotation_Matrix c;          //local axes of bod in pack config
            Coordinates t;              //pos of local axes of bod
            std::vector<tY9D9Z9K4> K4;  //elements in bod
            tY9D9Z9(){}
         };
         class tY9D9Q { /*local axes of potential shared bods in pack config*/
         public:
            Rotation_Matrix r;          //orientation of local axes
            Coordinates t;              //pos of origin
            tY9D9Q(){}
         };
      public:
         std::vector<tY9D9Z9> Z9;       //bods
         std::vector<tY9D9Q> Q;         //4 local coord systems
         double e[ 4];                  //energy of config [F,Fp,Fh,Fm]
         tY9D9():Q( 4){}
         void swap(tY9::tY9D9& a);
      };
   public:
      int oD9;                          //number
      std::vector<tY9D9> D9;            //set of configs
      int oZ9;                          //number of bods
      tY9(){}
      void operator=(const tY9& a);
      void operator=(const tQ9& a);
   };
//
//________________________________________________________________
   class tU9 { /*all subsystems containing oU bods*/
   public:
      class tU9Y9 { /*oU-bod subsystem*/
      public:
         class tU9Y9Z9 { /*bod of subsystem*/
         public:
            class tU9Y9Z9K4 { /*elem in bod*/
            public:
               short G7;                //index of elem
               char X;                  //x pos in sheet
               char Y;                  //y pos in sheet
               char sgnx;               //direction of CO and NH along x
               char sgny;               //direction of strand along y
               Coordinates u;           //pos of elem start in pack config
               Coordinates v;           //pos of elem end
               Rotation_Matrix r;       //local axes for elem in pack config
               tU9Y9Z9K4(){}
            };
         public:
            int S9;                     //index of bod
            int Y9;                     //index of (oU-1)-bod subsystem
            char X;                     //pos in subsystem
            float e;                    //energy of alignment left and right
            short A;                    //alignment wrt left bod
            short B;                    //alignment wrt right bod
            Rotation_Matrix c;          //local axes of bod in pack config
            Coordinates t;              //pos of local axes of bod
            int oK4;                    //number
            std::vector<tU9Y9Z9K4> K4;  //elements in bod
            tU9Y9Z9(){}
         };
         class tU9Y9Q { /*local axes of potential shared bods in pack config*/
         public:
            Rotation_Matrix r;          //orientation of local axes
            Coordinates t;              //pos of origin
            tU9Y9Q(){}
         };
      public:
         std::vector<tU9Y9Z9> Z9;       //bods of subsystem
         bool STABLE;                   //exists stable config
         bool FACTOR;                   //subsys +complement stable wrt union
         double e[4];                   //energy of config [F,Fp,Fh,Fm]
         std::vector<tU9Y9Q> Q;         //4 local coord systems
         tU9Y9():Q( 4){}
         void operator=(const tY9::tY9D9& a);
      };
   public:
      int oY9;                          //number
      std::vector<tU9Y9> Y9;            //oU-bod subsystems
      tU9():oY9( 0){}
      void operator=(const tU9& a);
   };
//
//________________________________________________________________
   class tT9 { /*factor subconfigs*/
   public:
      int U9;                   //number of bods
      int Y9;                   //optimal config
      bool sub;                 //in optimal union of subconfigs
      double e[4];              //SCORE of subconfig [F,Fp,Fh,Fm]
      tT9(){}
      void swap(tT9& a);
      void operator=(const tT9& a);
   };

private:
   int oH8;                                 //union of 0,1,2, and 3-bod patterns
   std::vector<tH8> H8;                     //hash of 0,1,2, or 3-bod patterns
   double alp0;                             //scale factor for SCORE0
   double alp1;                             //scale factor for SCORE1
   double alp1b;                            //scale factor for SCORE1b
   double alp2;                             //scale factor for SCORE2
   double alp2b;                            //scale factor for SCORE2b
   double alp3;                             //scale factor for SCORE3
   double alp3b;                            //scale factor for SCORE3b
   double alp3c;                            //scale factor for SCORE3c
   int iFOLD;                               //index of renormalization of SCORE
   double xFOLD;                            //folding coordinate [0=unf,1=f]
   Composition targ;                        //initial composition
   std::vector<double> o_R0C7e;             //residue ising model energies
   std::vector<double> o_R0C7C7e;           //
   std::vector<double> o_R0C7C7C7e;         //
   std::vector<double> o_R0C7C7C7C7e;       //
   std::vector<double> o_R0C7C7C7C7C7e;     //
   std::vector<double> o_R0C7C7C7C7C7C7e;   //
   std::vector<double> o_R0C7p;             //residue ising model probabilities
   std::vector<double> o_R0C7C7p;           //
   std::vector<double> o_R0C7C7C7p;         //
   std::vector<double> o_R0C7C7C7C7p;       //
   std::vector<double> o_R0C7C7C7C7C7p;     //
   std::vector<double> o_R0C7C7C7C7C7C7p;   //
   std::vector<double> o_R0C7C7C7C7C7C7a;   //path probabilities
   std::vector<int> o_R0mC7;                //lower bound of path
   std::vector<int> o_R0nC7;                //upper bound of path
   std::vector<double> o_R0C7C7C7C7C7z;     //partition func partial sums
   std::vector<double> o_R0C7C7C7C7C7q;     //
   std::vector<double> o_C7p;               //prob
   double g0;                               // G of res ising model
   double s0;                               // S of res ising model
   std::vector<short> o_R0R0v;              //bit filter
   std::vector<double> o_C7e;               //energy
   int oI7;                                 //size
   std::vector<tK7> I7;                     //high SCORE states, equiv T7 seq
   int iCYCLE;                              //index of cycle through regions
   bool SWITCH;                             //cycle includes change to zSCORE
   int i0R0;                                //first residue of region
   int i7R0;                                //last residue of region
   int oG7;                                 //dimension of elements in chain
   std::vector<tG7> G7;                     //elements
   std::vector<bool> o_T7T7T7T7T7T7T7b;     //composition has been scored
   int mG7grp;                              //first element of region
   int nG7grp;                              //last element of region
   bool NEWGRP;                             //first evaluation for new group
   std::vector<int> o_G7jR0;                //end res of elem
   std::vector<int> o_G7kR0;                //end res of elem
   std::vector<double> o_G7C7d;             //element ising model energies
   std::vector<double> o_G7C7C7d;           //
   std::vector<double> o_G7C7C7C7d;         //
   std::vector<double> o_G7C7C7C7C7d;       //
   std::vector<double> o_G7C7C7C7C7C7d;     //
   std::vector<double> o_G7C7C7C7C7C7C7d;   //
   std::vector<double> o_G7C7o;             //element ising model probabilities
   std::vector<double> o_G7C7C7o;           //
   std::vector<double> o_G7C7C7C7o;         //
   std::vector<double> o_G7C7C7C7C7o;       //
   std::vector<double> o_G7C7C7C7C7C7o;     //
   std::vector<double> o_G7C7C7C7C7C7C7o;   //
   std::vector<double> o_G7C7C7C7C7C7C7b;   //path probabilities wo res imp
   std::vector<int> o_G7mC7;                //lower bound of path
   std::vector<int> o_G7nC7;                //upper bound of path
   std::vector<double> o_G7C7C7C7C7C7y;     //partition func partial sums
   std::vector<double> o_G7C7C7C7C7C7r;     //
   double g2;                               // G of igor model with elem imp
   double s2;                               // S of igor model with elem imp
   std::vector<int> o_R0G7;                 //index of elem
   std::vector<double> o_G7C7e;             //element ising model energies
   std::vector<double> o_G7C7C7e;           //
   std::vector<double> o_G7C7C7C7e;         //
   std::vector<double> o_G7C7C7C7C7e;       //
   std::vector<double> o_G7C7C7C7C7C7e;     //
   std::vector<double> o_G7C7C7C7C7C7C7e;   //
   std::vector<double> o_G7C7p;             //element ising model probabilities
   std::vector<double> o_G7C7C7p;           //
   std::vector<double> o_G7C7C7C7p;         //
   std::vector<double> o_G7C7C7C7C7p;       //
   std::vector<double> o_G7C7C7C7C7C7p;     //
   std::vector<double> o_G7C7C7C7C7C7C7p;   //
   std::vector<double> o_G7C7C7C7C7C7C7a;   //path probabilities with res imp
   std::vector<double> o_G7C7C7C7C7C7z;     //partition func partial sums
   std::vector<double> o_G7C7C7C7C7C7q;     //
   double g1;                               // G of igor model with elem+res imp
   double s1;                               // S of igor model with elem+res imp
   int oE8;                                 //n of strands from res ising model
   int nF8;                                 //n bits of index space for combos
   int oF8;                                 //n of bound combos per strand
   std::vector<int> o_E8F8rr;               //map of E8xF8 to R0xR0
   std::vector<short> o_R0R0ef;             //map of R0xR0 to E8xF8
   std::vector<int> o_E8F8Q8;               //map {E8-->F8} to backup space
   int oQ8;                                 //number of sheet compositions
   std::vector<tQ8> Q8;                     //backup space
   int oE9;                                 //n of helices from res ising model
   int nF9;                                 //n bits of index space for combos
   int oF9;                                 //n of bound combos per helix
   std::vector<int> o_E9F9rr;               //map of E9xF9 to R0xR0
   std::vector<short> o_R0R0hf;             //map of R0xR0 to E9xF9
   std::vector<int> o_E9F9Q9;               //map {E9-->F9} to backup space
   int oQ9;                                 //number of pack compositions
   std::vector<tQ9> Q9;                     //backup space
   bool ALTPATH;                            //search alternative paths
   bool NATIVE;                             //filter output
   int oW4;                                 //number
   std::vector<tW4> W4;                     //hydrophob patches on bods
   int oM8;                                 //number
   std::vector<tM8> M8;                     //collection of strands isolated
   int oU8;                                 //number
   std::vector<tU8> U8;                     //n of strands in subsystem
   int pU8;                                 //index
   int oY8;                                 //number
   std::vector<tY8> Y8;                     //oU-strand subsystems
   std::vector<tZ8> Z8;                     //strands of sheet
   bool PARALLEL;                           //contains adjacent parallel strands
   double h;                                //measure of sheet polarity
   int oI6;                                 //number
   std::vector<tY8> I6;                     //oU-strand subsystems
   int mcha;                                //n of residues separating strands
   int oT8;                                 //number
   std::vector<tT8> T8;                     //factor subconfigs
   int oV8;                                 //number
   std::vector<tV8> V8;                     //sheet of union of subconfigs
   int oM9;                                 //number
   std::vector<tM9> M9;                     //collection of bods isolated
   int oU9;                                 //number
   std::vector<tU9> U9;                     //n of bods in subsystem
   int pU9;                                 //index
   int oY9;                                 //number
   std::vector<tY9> Y9;                     //oU-bod subsystems
   int oI9;                                 //number
   std::vector<tY9> I9;                     //oU-bod subsystems
   std::vector<tZ9> Z9;                     //strands of sheet
   int oT9;                                 //number
   std::vector<tT9> T9;                     //factor subconfigs
   double SCORE;                            //total score
   double SCORE0;                           //ln( probability of composition)
   double SCORE1;                           //( g1-s1) -( g0-s0)
   double SCORE1b;                          //( s2-s1)
   double SCORE2;                           //-g2
   double SCORE2b;                          // s2
   double SCORE3;                           //formation of bods +packing of bods
   double SCORE3b;                          // Fh
   double SCORE3c;                          // Fm
   int oM7;                                 //size
   std::vector<tK7> M7;                     //high SCORE states, partly folded
   std::vector<double> o_XXXXXeps;          //{ 1,-1} st n-linear +alternating
   std::vector<double> o_CM;                //Cayley-Menger 5x5 matrix

   double& R0C7e(int i,int a){
      return o_R0C7e.at( i*3 +(a-1));
   }
   double& R0C7C7e(int i,int a,int b){
      return o_R0C7C7e.at( i*9 +(a-1)*3 +(b-1));
   }
   double& R0C7C7C7e(int i,int a,int b,int c){
      return o_R0C7C7C7e.at( i*27 +(a-1)*9 +(b-1)*3 +(c-1));
   }
   double& R0C7C7C7C7e(int i,int a,int b,int c,int d){
      return o_R0C7C7C7C7e.at( i*81 +(a-1)*27 +(b-1)*9 +(c-1)*3 +(d-1));
   }
   double& R0C7C7C7C7C7e(int i,int a,int b,int c,int d,int e){
      return o_R0C7C7C7C7C7e.at( i*243 +(a-1)*81 +(b-1)*27 +(c-1)*9 +(d-1)*3
        +(e-1));
   }
   double& R0C7C7C7C7C7C7e(int i,int a,int b,int c,int d,int e,int f){
      return o_R0C7C7C7C7C7C7e.at( i*729 +(a-1)*243 +(b-1)*81 +(c-1)*27
        +(d-1)*9 +(e-1)*3 +(f-1));
   }
   double& R0C7p(int i,int a){
      return o_R0C7p.at( i*3 +(a-1));
   }
   double& R0C7C7p(int i,int a,int b){
      return o_R0C7C7p.at( i*9 +(a-1)*3 +(b-1));
   }
   double& R0C7C7C7p(int i,int a,int b,int c){
      return o_R0C7C7C7p.at( i*27 +(a-1)*9 +(b-1)*3 +(c-1));
   }
   double& R0C7C7C7C7p(int i,int a,int b,int c,int d){
      return o_R0C7C7C7C7p.at( i*81 +(a-1)*27 +(b-1)*9 +(c-1)*3 +(d-1));
   }
   double& R0C7C7C7C7C7p(int i,int a,int b,int c,int d,int e){
      return o_R0C7C7C7C7C7p.at( i*243 +(a-1)*81 +(b-1)*27 +(c-1)*9 +(d-1)*3
        +(e-1));
   }
   double& R0C7C7C7C7C7C7p(int i,int a,int b,int c,int d,int e,int f){
      return o_R0C7C7C7C7C7C7p.at( i*729 +(a-1)*243 +(b-1)*81 +(c-1)*27
        +(d-1)*9 +(e-1)*3 +(f-1));
   }
   double& R0C7C7C7C7C7C7a(int i,int a,int b,int c,int d,int e,int f){
      return o_R0C7C7C7C7C7C7a.at( i*4096 +a*1024 +b*256 +c*64 +d*16 +e*4 +f);
   }
   int& R0mC7(int i){
      return o_R0mC7.at(i+5);
   }
   int& R0nC7(int i){
      return o_R0nC7.at(i+5);
   }
   double& R0C7C7C7C7C7z(int i,int a,int b,int c,int d,int e){
      return o_R0C7C7C7C7C7z.at( (i+1)*1024 +a*256 +b*64 +c*16 +d*4 +e);
   }
   double& R0C7C7C7C7C7q(int i,int a,int b,int c,int d,int e){
      return o_R0C7C7C7C7C7q.at( (i+1)*1024 +a*256 +b*64 +c*16 +d*4 +e);
   }
   double& C7p(int a){
      return o_C7p.at(a-1);
   }
   short& R0R0v(int i,int j){
      return o_R0R0v.at( i*oR0 -i*(i+1)/2 +j);
   }
   double& C7e(int a){
      return o_C7e.at(a-1);
   }
   bool T7T7T7T7T7T7T7b(int a,int b,int c,int d,int e,int f,int g){
      return o_T7T7T7T7T7T7T7b[ (a-1)*24137569 +b*1419857 +c*83521 +d*4913
                               +e*289 +f*17 +g];
   }
   void T7T7T7T7T7T7T7b(int a,int b,int c,int d,int e,int f,int g,bool h){
      o_T7T7T7T7T7T7T7b[ (a-1)*24137569 +b*1419857 +c*83521 +d*4913
                        +e*289 +f*17 +g]=h;
      return;
   }
   int& G7jR0(int i){
      return o_G7jR0.at(i+1);
   }
   int& G7kR0(int i){
      return o_G7kR0.at(i+1);
   }
   double& G7C7d(int i,int a){
      return o_G7C7d.at( i*3 +(a-1));
   }
   double& G7C7C7d(int i,int a,int b){
      return o_G7C7C7d.at( i*9 +(a-1)*3 +(b-1));
   }
   double& G7C7C7C7d(int i,int a,int b,int c){
      return o_G7C7C7C7d.at( i*27 +(a-1)*9 +(b-1)*3 +(c-1));
   }
   double& G7C7C7C7C7d(int i,int a,int b,int c,int d){
      return o_G7C7C7C7C7d.at( i*81 +(a-1)*27 +(b-1)*9 +(c-1)*3 +(d-1));
   }
   double& G7C7C7C7C7C7d(int i,int a,int b,int c,int d,int e){
      return o_G7C7C7C7C7C7d.at( i*243 +(a-1)*81 +(b-1)*27 +(c-1)*9 +(d-1)*3
        +(e-1));
   }
   double& G7C7C7C7C7C7C7d(int i,int a,int b,int c,int d,int e,int f){
      return o_G7C7C7C7C7C7C7d.at( i*729 +(a-1)*243 +(b-1)*81 +(c-1)*27
        +(d-1)*9 +(e-1)*3 +(f-1));
   }
   double& G7C7o(int i,int a){
      return o_G7C7o.at( i*3 +(a-1));
   }
   double& G7C7C7o(int i,int a,int b){
      return o_G7C7C7o.at( i*9 +(a-1)*3 +(b-1));
   }
   double& G7C7C7C7o(int i,int a,int b,int c){
      return o_G7C7C7C7o.at( i*27 +(a-1)*9 +(b-1)*3 +(c-1));
   }
   double& G7C7C7C7C7o(int i,int a,int b,int c,int d){
      return o_G7C7C7C7C7o.at( i*81 +(a-1)*27 +(b-1)*9 +(c-1)*3 +(d-1));
   }
   double& G7C7C7C7C7C7o(int i,int a,int b,int c,int d,int e){
      return o_G7C7C7C7C7C7o.at( i*243 +(a-1)*81 +(b-1)*27 +(c-1)*9 +(d-1)*3
        +(e-1));
   }
   double& G7C7C7C7C7C7C7o(int i,int a,int b,int c,int d,int e,int f){
      return o_G7C7C7C7C7C7C7o.at( i*729 +(a-1)*243 +(b-1)*81 +(c-1)*27
        +(d-1)*9 +(e-1)*3 +(f-1));
   }
   double& G7C7C7C7C7C7C7a(int i,int a,int b,int c,int d,int e,int f){
      return o_G7C7C7C7C7C7C7a.at( i*4096 +a*1024 +b*256 +c*64 +d*16 +e*4 +f);
   }
   int& G7mC7(int i){
      return o_G7mC7.at(i+5);
   }
   int& G7nC7(int i){
      return o_G7nC7.at(i+5);
   }
   double& G7C7C7C7C7C7y(int i,int a,int b,int c,int d,int e){
      return o_G7C7C7C7C7C7y.at( (i+1)*1024 +a*256 +b*64 +c*16 +d*4 +e);
   }
   double& G7C7C7C7C7C7r(int i,int a,int b,int c,int d,int e){
      return o_G7C7C7C7C7C7r.at( (i+1)*1024 +a*256 +b*64 +c*16 +d*4 +e);
   }
   int& R0G7(int i){
      return o_R0G7.at( i);
   }
   double& G7C7e(int i,int a){
      return o_G7C7e.at( i*3 +(a-1));
   }
   double& G7C7C7e(int i,int a,int b){
      return o_G7C7C7e.at( i*9 +(a-1)*3 +(b-1));
   }
   double& G7C7C7C7e(int i,int a,int b,int c){
      return o_G7C7C7C7e.at( i*27 +(a-1)*9 +(b-1)*3 +(c-1));
   }
   double& G7C7C7C7C7e(int i,int a,int b,int c,int d){
      return o_G7C7C7C7C7e.at( i*81 +(a-1)*27 +(b-1)*9 +(c-1)*3 +(d-1));
   }
   double& G7C7C7C7C7C7e(int i,int a,int b,int c,int d,int e){
      return o_G7C7C7C7C7C7e.at( i*243 +(a-1)*81 +(b-1)*27 +(c-1)*9 +(d-1)*3
        +(e-1));
   }
   double& G7C7C7C7C7C7C7e(int i,int a,int b,int c,int d,int e,int f){
      return o_G7C7C7C7C7C7C7e.at( i*729 +(a-1)*243 +(b-1)*81 +(c-1)*27
        +(d-1)*9 +(e-1)*3 +(f-1));
   }
   double& G7C7p(int i,int a){
      return o_G7C7p.at( i*3 +(a-1));
   }
   double& G7C7C7p(int i,int a,int b){
      return o_G7C7C7p.at( i*9 +(a-1)*3 +(b-1));
   }
   double& G7C7C7C7p(int i,int a,int b,int c){
      return o_G7C7C7C7p.at( i*27 +(a-1)*9 +(b-1)*3 +(c-1));
   }
   double& G7C7C7C7C7p(int i,int a,int b,int c,int d){
      return o_G7C7C7C7C7p.at( i*81 +(a-1)*27 +(b-1)*9 +(c-1)*3 +(d-1));
   }
   double& G7C7C7C7C7C7p(int i,int a,int b,int c,int d,int e){
      return o_G7C7C7C7C7C7p.at( i*243 +(a-1)*81 +(b-1)*27 +(c-1)*9 +(d-1)*3
        +(e-1));
   }
   double& G7C7C7C7C7C7C7p(int i,int a,int b,int c,int d,int e,int f){
      return o_G7C7C7C7C7C7C7p.at( i*729 +(a-1)*243 +(b-1)*81 +(c-1)*27
        +(d-1)*9 +(e-1)*3 +(f-1));
   }
   double& G7C7C7C7C7C7C7b(int i,int a,int b,int c,int d,int e,int f){
      return o_G7C7C7C7C7C7C7b.at( i*4096 +a*1024 +b*256 +c*64 +d*16 +e*4 +f);
   }
   double& G7C7C7C7C7C7z(int i,int a,int b,int c,int d,int e){
      return o_G7C7C7C7C7C7z.at( (i+1)*1024 +a*256 +b*64 +c*16 +d*4 +e);
   }
   double& G7C7C7C7C7C7q(int i,int a,int b,int c,int d,int e){
      return o_G7C7C7C7C7C7q.at( (i+1)*1024 +a*256 +b*64 +c*16 +d*4 +e);
   }
   int& E8F8rr(int i,int j){
      return o_E8F8rr.at( i*oF8 +j);
   }
   short& R0R0ef(int i,int j){
      return o_R0R0ef.at( i*oR0 -i*(i+1)/2 +j);
   }
   int& E8F8Q8(int i){
      return o_E8F8Q8.at( i);
   }
   int& E9F9rr(int i,int j){
      return o_E9F9rr.at( i*oF9 +j);
   }
   short& R0R0hf(int i,int j){
      return o_R0R0hf.at( i*oR0 -i*(i+1)/2 +j);
   }
   int& E9F9Q9(int i){
      return o_E9F9Q9.at( i);
   }
   double& XXXXXeps(int a,int b,int c,int d,int e){
      return o_XXXXXeps.at( a*625 +b*125 +c*25 +d*5 +e);
   }
   double& CM(int i,int j){
      return o_CM.at( i*5 +j);
   }

   void IGO_EHASH(const DAT_IGOR_DATA& igor_data);
   void IGO_RLOAD(const DAT_IGOR_DATA& igor_data);
   void IGO_RLOAD_1(const DAT_IGOR_DATA& igor_data,
                    int iR0,
                    int iH8);
   void IGO_RPROB(const DAT_IGOR_DATA& igor_data,
                  Output_Streams& out);
   void IGO_ULOAD(const DAT_IGOR_DATA& igor_data,
                  Output_Streams& out);
   void IGO_GFILT(const DAT_PHYSICS_CONSTS& physics_consts,
                  const DAT_IGOR_DATA& igor_data,
                  Output_Streams& out);
   void IGO_BHASH(const DAT_IGOR_DATA& igor_data);
   void IGO_2TRAJ(const DAT_PHYSICS_CONSTS& physics_consts,
                  const DAT_IGOR_DATA& igor_data,
                  Output_Streams& out);
   void IGO_GSEQ(const DAT_IGOR_DATA& igor_data);
   void IGO_GLOADB(const DAT_IGOR_DATA& igor_data);
   void IGO_GLOADB_1(const DAT_IGOR_DATA& igor_data,
                     int iG7,
                     int iH8);
   void IGO_SCO2(const DAT_IGOR_DATA& igor_data,
                 Output_Streams& out);
   void IGO_GLOADE(const DAT_IGOR_DATA& igor_data);
   void IGO_SCO1(const DAT_IGOR_DATA& igor_data,
                 Output_Streams& out);
   void IGO_SCO0(const DAT_IGOR_DATA& igor_data);
   void IGO_STEP(const DAT_PHYSICS_CONSTS& physics_consts,
                 const DAT_IGOR_DATA& igor_data,
                 Output_Streams& out);
   void IGO_3TRAJ(const DAT_PHYSICS_CONSTS& physics_consts,
                  const DAT_ARRAY_CONSTS& array_consts,
                  const DAT_IGOR_DATA& igor_data,
                  Output_Streams& out,
                  const Structure& str);
   void IGO_3TRAJ_B();
   void IGO_SCO3(const DAT_PHYSICS_CONSTS& physics_consts,
                 const DAT_ARRAY_CONSTS& array_consts,
                 const DAT_IGOR_DATA& igor_data,
                 Output_Streams& out);
   void IGO_SCO3_E(const DAT_PHYSICS_CONSTS& physics_consts,
                   const DAT_ARRAY_CONSTS& array_consts,
                   const DAT_IGOR_DATA& igor_data,
                   Output_Streams& out,
                   int iM8);
   bool IGO_SCO3_E1(const DAT_PHYSICS_CONSTS& physics_consts,
                    const DAT_IGOR_DATA& igor_data,
                    Output_Streams& out,
                    int iM8);
   bool IGO_SCO3_E2(const DAT_PHYSICS_CONSTS& physics_consts,
                    const DAT_ARRAY_CONSTS& array_consts,
                    const DAT_IGOR_DATA& igor_data,
                    Output_Streams& out,
                    int iM8);
   bool IGO_SCO3_EDOC(Output_Streams& out,
                      int pZ8,int lZ8,int lX,int lY);
   int IGO_SCO3_EPCK(const DAT_PHYSICS_CONSTS& physics_consts,
                     const DAT_IGOR_DATA& igor_data,
                     Output_Streams& out,
                     int iM8,int iY8,
                     int lX,int uX,int lY,int uY,
                     int lYup,int sgn);
   Coordinates IGO_SCO3_EPEN(const DAT_PHYSICS_CONSTS& physics_consts,
                             const DAT_IGOR_DATA& igor_data,
                             Output_Streams& out,
                             int iM8,int iY8,
                             std::vector<char>& o_Z8A,
                             std::vector<char>& o_Z8B,
                             std::vector<int>& o_XZ8);
   void IGO_SCO3_EVIS(const DAT_IGOR_DATA& igor_data,
                      Output_Streams& out,
                      int iM8,int iY8,
                      int lX,int uX,int lY,int uY);
   void IGO_SCO3_OPTV(const DAT_IGOR_DATA& igor_data,
                      Output_Streams& out,
                      int iM8,int iU8,int iY8,
                      int lX,int uX,int lY,int uY);
   void IGO_SCO3_OPTW(const DAT_IGOR_DATA& igor_data,
                      int iM8,int iU8,int iY8,
                      int lX,int uX,int lY,int uY,
                      int lYup,int sgn,int upen);
   void IGO_SCO3_W(const DAT_PHYSICS_CONSTS& physics_consts,
                   const DAT_ARRAY_CONSTS& array_consts,
                   const DAT_IGOR_DATA& igor_data,
                   Output_Streams& out);
   bool IGO_SCO3_W1(const DAT_PHYSICS_CONSTS& physics_consts,
                    const DAT_IGOR_DATA& igor_data,
                    Output_Streams& out);
   bool IGO_SCO3_W2(const DAT_PHYSICS_CONSTS& physics_consts,
                    const DAT_ARRAY_CONSTS& array_consts,
                    const DAT_IGOR_DATA& igor_data,
                    Output_Streams& out);
   bool IGO_SCO3_W1DOC(const DAT_PHYSICS_CONSTS& physics_consts,
                       const DAT_IGOR_DATA& igor_data,
                       Output_Streams& out,
                       int iW4,int jW4,int iC8,
                       double& e,
                       bool OPTIMIZE);
   bool IGO_SCO3_W2DOC(const DAT_PHYSICS_CONSTS& physics_consts,
                       const DAT_IGOR_DATA& igor_data,
                       Output_Streams& out,
                       int iY9,
                       int aY9,int aD9,int bY9,int bD9,
                       int pZ9,int lZ9,
                       int pS9,int lS9,
                       int aX,int bX,
                       std::vector<int>& o_XZ9,
                       std::vector<short>& o_Z9A,
                       std::vector<short>& o_Z9B,
                       std::vector<float>& o_Z9e,double& e,
                       Coordinates& t,Coordinates& u,
                       Rotation_Matrix& r);
   bool IGO_SCO3_WOLP(const DAT_PHYSICS_CONSTS& physics_consts,
                      Output_Streams& out,
                      int pZ9,int lZ9);
   bool IGO_SCO3_WPOL(const DAT_PHYSICS_CONSTS& physics_consts,
                      const DAT_IGOR_DATA& igor_data,
                      Output_Streams& out,
                      double& Fm,double& Fh);
   bool IGO_3TRAJ_X(const DAT_PHYSICS_CONSTS& physics_consts,
                    Output_Streams& out);
   double IGO_CAYLEY_MENGER(Output_Streams& out,
                            bool LOWROOT);

public:
   int oR0;                                 //n
   std::vector<tR0> R0;                     //set of residues in chain
   int oK7;                                 //size
   std::vector<tK7> K7;                     //high SCORE chain states
   double zSCORE;                           //highest score over trajectory
   double zSCORE0;                          //components of zSCORE
   double zSCORE1;                          //
   double zSCORE1b;                         //
   double zSCORE2;                          //
   double zSCORE2b;                         //
   double zSCORE3;                          //
   double zSCORE3b;                         //
   double zSCORE3c;                         //
   int zoG7;                                //number
   std::vector<tG7> zG7;                    //elements optimal
   int zoW4;                                //number
   std::vector<tW4> zW4;                    //hydrophob patches on elems
   int zoM8;                                //number
   std::vector<tM8> zM8;                    //collection of strands isolated
   int zoU8;                                //number
   std::vector<tU8> zU8;                    //oU-strand subsystems
   int zoT8;                                //number
   std::vector<tT8> zT8;                    //factor subconfigs
   int zoV8;                                //number
   std::vector<tV8> zV8;                    //sheet of union of subconfigs
   int zoM9;                                //number
   std::vector<tM9> zM9;                    //collection of bods isolated
   int zoU9;                                //number
   std::vector<tU9> zU9;                    //oU-bod subsystems
   int zoT9;                                //number
   std::vector<tT9> zT9;                    //factor subconfigs

   void IGO(const DAT_PHYSICS_CONSTS& physics_consts,
            const DAT_ARRAY_CONSTS& array_consts,
            const DAT_RESIDUE_MAPPINGS& residue_mappings,
            const DAT_IGOR_DATA& igor_data,
            const Thread_Options& opt,
            Output_Streams& out,
            const Structure& str,
            int pZ0,
            int pR0min,
            int pR0max);

   Igor_Model():
      oH8(51357),
      H8(oH8),
      o_XXXXXeps(3125, (0.00)),
      o_CM(25, (0.00))
   {
//
      for(int i=1;i<5;i++){
         CM(i,0)= (1.00);
         CM(0,i)= (1.00);
      }
//
// 0 order changes
      XXXXXeps(0,1,2,3,4)=(  1.00);
      XXXXXeps(4,0,1,2,3)=(  1.00);
      XXXXXeps(3,4,0,1,2)=(  1.00);
      XXXXXeps(2,3,4,0,1)=(  1.00);
      XXXXXeps(1,2,3,4,0)=(  1.00);
//
// 1 order changes
      XXXXXeps(1,0,2,3,4)=( -1.00);
      XXXXXeps(4,1,0,2,3)=( -1.00);
      XXXXXeps(3,4,1,0,2)=( -1.00);
      XXXXXeps(2,3,4,1,0)=( -1.00);
      XXXXXeps(0,2,3,4,1)=( -1.00);
//
      XXXXXeps(2,1,0,3,4)=( -1.00);
      XXXXXeps(4,2,1,0,3)=( -1.00);
      XXXXXeps(3,4,2,1,0)=( -1.00);
      XXXXXeps(0,3,4,2,1)=( -1.00);
      XXXXXeps(1,0,3,4,2)=( -1.00);
//
      XXXXXeps(3,1,2,0,4)=( -1.00);
      XXXXXeps(4,3,1,2,0)=( -1.00);
      XXXXXeps(0,4,3,1,2)=( -1.00);
      XXXXXeps(2,0,4,3,1)=( -1.00);
      XXXXXeps(1,2,0,4,3)=( -1.00);
//
      XXXXXeps(4,1,2,3,0)=( -1.00);
      XXXXXeps(0,4,1,2,3)=( -1.00);
      XXXXXeps(3,0,4,1,2)=( -1.00);
      XXXXXeps(2,3,0,4,1)=( -1.00);
      XXXXXeps(1,2,3,0,4)=( -1.00);
//
      XXXXXeps(0,2,1,3,4)=( -1.00);
      XXXXXeps(4,0,2,1,3)=( -1.00);
      XXXXXeps(3,4,0,2,1)=( -1.00);
      XXXXXeps(1,3,4,0,2)=( -1.00);
      XXXXXeps(2,1,3,4,0)=( -1.00);
//
      XXXXXeps(0,3,2,1,4)=( -1.00);
      XXXXXeps(4,0,3,2,1)=( -1.00);
      XXXXXeps(1,4,0,3,2)=( -1.00);
      XXXXXeps(2,1,4,0,3)=( -1.00);
      XXXXXeps(3,2,1,4,0)=( -1.00);
//
      XXXXXeps(0,4,2,3,1)=( -1.00);
      XXXXXeps(1,0,4,2,3)=( -1.00);
      XXXXXeps(3,1,0,4,2)=( -1.00);
      XXXXXeps(2,3,1,0,4)=( -1.00);
      XXXXXeps(4,2,3,1,0)=( -1.00);
//
      XXXXXeps(0,1,3,2,4)=( -1.00);
      XXXXXeps(4,0,1,3,2)=( -1.00);
      XXXXXeps(2,4,0,1,3)=( -1.00);
      XXXXXeps(3,2,4,0,1)=( -1.00);
      XXXXXeps(1,3,2,4,0)=( -1.00);
//
      XXXXXeps(0,1,4,3,2)=( -1.00);
      XXXXXeps(2,0,1,4,3)=( -1.00);
      XXXXXeps(3,2,0,1,4)=( -1.00);
      XXXXXeps(4,3,2,0,1)=( -1.00);
      XXXXXeps(1,4,3,2,0)=( -1.00);
//
      XXXXXeps(0,1,2,4,3)=( -1.00);
      XXXXXeps(3,0,1,2,4)=( -1.00);
      XXXXXeps(4,3,0,1,2)=( -1.00);
      XXXXXeps(2,4,3,0,1)=( -1.00);
      XXXXXeps(1,2,4,3,0)=( -1.00);
//
// 2 order changes
      XXXXXeps(2,0,1,3,4)=(  1.00);
      XXXXXeps(4,2,0,1,3)=(  1.00);
      XXXXXeps(3,4,2,0,1)=(  1.00);
      XXXXXeps(1,3,4,2,0)=(  1.00);
      XXXXXeps(0,1,3,4,2)=(  1.00);
//
      XXXXXeps(3,0,2,1,4)=(  1.00);
      XXXXXeps(4,3,0,2,1)=(  1.00);
      XXXXXeps(1,4,3,0,2)=(  1.00);
      XXXXXeps(2,1,4,3,0)=(  1.00);
      XXXXXeps(0,2,1,4,3)=(  1.00);
//
      XXXXXeps(4,0,2,3,1)=(  1.00);
      XXXXXeps(1,4,0,2,3)=(  1.00);
      XXXXXeps(3,1,4,0,2)=(  1.00);
      XXXXXeps(2,3,1,4,0)=(  1.00);
      XXXXXeps(0,2,3,1,4)=(  1.00);
//
      XXXXXeps(3,1,0,2,4)=(  1.00);
      XXXXXeps(4,3,1,0,2)=(  1.00);
      XXXXXeps(2,4,3,1,0)=(  1.00);
      XXXXXeps(0,2,4,3,1)=(  1.00);
      XXXXXeps(1,0,2,4,3)=(  1.00);
//
      XXXXXeps(4,1,0,3,2)=(  1.00);
      XXXXXeps(2,4,1,0,3)=(  1.00);
      XXXXXeps(3,2,4,1,0)=(  1.00);
      XXXXXeps(0,3,2,4,1)=(  1.00);
      XXXXXeps(1,0,3,2,4)=(  1.00);
//
      XXXXXeps(2,1,0,4,3)=(  1.00);
      XXXXXeps(3,2,1,0,4)=(  1.00);
      XXXXXeps(4,3,2,1,0)=(  1.00);
      XXXXXeps(0,4,3,2,1)=(  1.00);
      XXXXXeps(1,0,4,3,2)=(  1.00);
//
      XXXXXeps(4,1,2,0,3)=(  1.00);
      XXXXXeps(3,4,1,2,0)=(  1.00);
      XXXXXeps(0,3,4,1,2)=(  1.00);
      XXXXXeps(2,0,3,4,1)=(  1.00);
      XXXXXeps(1,2,0,3,4)=(  1.00);
//
      XXXXXeps(0,3,1,2,4)=(  1.00);
      XXXXXeps(4,0,3,1,2)=(  1.00);
      XXXXXeps(2,4,0,3,1)=(  1.00);
      XXXXXeps(1,2,4,0,3)=(  1.00);
      XXXXXeps(3,1,2,4,0)=(  1.00);
//
      XXXXXeps(0,4,1,3,2)=(  1.00);
      XXXXXeps(2,0,4,1,3)=(  1.00);
      XXXXXeps(3,2,0,4,1)=(  1.00);
      XXXXXeps(1,3,2,0,4)=(  1.00);
      XXXXXeps(4,1,3,2,0)=(  1.00);
//
      XXXXXeps(0,4,2,1,3)=(  1.00);
      XXXXXeps(3,0,4,2,1)=(  1.00);
      XXXXXeps(1,3,0,4,2)=(  1.00);
      XXXXXeps(2,1,3,0,4)=(  1.00);
      XXXXXeps(4,2,1,3,0)=(  1.00);
//
      XXXXXeps(0,1,4,2,3)=(  1.00);
      XXXXXeps(3,0,1,4,2)=(  1.00);
      XXXXXeps(2,3,0,1,4)=(  1.00);
      XXXXXeps(4,2,3,0,1)=(  1.00);
      XXXXXeps(1,4,2,3,0)=(  1.00);
//
// 3 order changes
      XXXXXeps(2,0,3,1,4)=( -1.00);
      XXXXXeps(4,2,0,3,1)=( -1.00);
      XXXXXeps(1,4,2,0,3)=( -1.00);
      XXXXXeps(3,1,4,2,0)=( -1.00);
      XXXXXeps(0,3,1,4,2)=( -1.00);
//
      XXXXXeps(3,0,2,4,1)=( -1.00);
      XXXXXeps(1,3,0,2,4)=( -1.00);
      XXXXXeps(4,1,3,0,2)=( -1.00);
      XXXXXeps(2,4,1,3,0)=( -1.00);
      XXXXXeps(0,2,4,1,3)=( -1.00);
   }
};

bool operator==(const Igor_Model::Composition& a,
                const Igor_Model::Composition& b);
bool operator<(const Igor_Model::Composition& a,
               const Igor_Model::Composition& b);
bool operator==(const Igor_Model::tK7& a,
                const Igor_Model::tK7& b);
bool bcomp(const Igor_Model::tK7& a,
           const Igor_Model::tK7& b);
int dcomp(const Igor_Model::tK7& a,
          const Igor_Model::tK7& b);
std::ostream& operator<<(std::ostream& os,
                         const std::vector<Igor_Model::tK7>& a);
std::ostream& operator<<(std::ostream& os,
                         const std::vector<Igor_Model::tW4>& a);
std::ostream& operator<<(std::ostream& os,
                         const std::vector<Igor_Model::tM8>& a);
std::ostream& operator<<(std::ostream& os,
                         const std::vector<Igor_Model::tU8>& a);
std::ostream& operator<<(std::ostream& os,
                         const std::vector<Igor_Model::tT8>& a);
std::ostream& operator<<(std::ostream& os,
                         const std::vector<Igor_Model::tV8>& a);
std::ostream& operator<<(std::ostream& os,
                         const std::vector<Igor_Model::tM9>& a);
std::ostream& operator<<(std::ostream& os,
                         const std::vector<Igor_Model::tU9>& a);
std::ostream& operator<<(std::ostream& os,
                         const std::vector<Igor_Model::tT9>& a);

#endif
