#ifndef DEF_GLO_AUTOMATIC
#define DEF_GLO_AUTOMATIC

#include "../glo/Backbone_Defs.hh"
#include "../phi/Coordinates.hh"
#include "../phi/Multipoles.hh"
#include "../phi/Rotation_Matrix.hh"
#include <string>
#include <vector>

class Glo_Automatic {
public:
   class tZ0 { /*chains in system of molecules*/
   public:
      int GENa;                 //start index into forward subgroups
      int cGEN;                 //number of forward subgroups
      tZ0(){}
   };
   class tQ1 { /*torsions in chains [forward order]*/
   public:
      Rotation_Matrix tu;       //t(chi about x-axis)u(lam about z-axis)
      int Q2;                   //map to forward variable torsion
      tQ1(){}
   };
   class tF1 { /*atoms in chains [backward order]*/
   public:
      Coordinates x;            //coords (bohr)
      Multipoles q;             //atom multipoles
      tF1(){}
   };
   class tG1 { /*atoms in chain forward groups [forward order]*/
   public:
      Coordinates b;            //untransformed coords (bohr)
      tG1(){}
   };
   class tQ2 { /*torsions in torsion-contracted chains [forward order]*/
   public:
      int Q1;                   //map to forward torsion
      int ORD;                  //map to order of generation
      int fZ2;                  //segment containing forward group
      int bZ2;                  //segment containing backward group
      tQ2():fZ2(-1),bZ2(-1){}
   };
   class tGEN { /*generated groups*/
   public:
      int Q1;                   //map to forward torsion
      tGEN(){}
   };
   class tG6 { /*generated residues used in deformation screen*/
   public:
      int Z2;                   //mapping of generated res to segment
      tG6():Z2(-1){}
   };
   class tD2 { /*combined backbone deformations*/
   public:
      std::vector<Backbone_Defs::tR1> R1;       //sequence of residue confs
      std::vector<int> o_Z2iD1;                 //initial lattice conf
      std::vector<double> o_U2chi;              //(degree) off lattice conf
      int star;                                 //num in star region
      int bord;                                 //num outside low-energy region
      double f[7];                              //screen energy and components
      int& Z2iD1(int i){
         return o_Z2iD1.at( i);
      }
      double& U2chi(int i){
         return o_U2chi.at( i);
      }
      const double& U2chi(int i) const {
         return o_U2chi.at( i);
      }
      tD2(int oR1,int oZ2,int oU2):
         R1(oR1),
         o_Z2iD1(oZ2),
         o_U2chi(oU2)
      {
      }
      void operator=(tD2& a);
      void swap(tD2& a);
   };
   class tZ2 { /*deformable segments*/
   public:
      int nD1;                          //num of backbone defs
      std::vector<double> o_D1e;        //lattice 1-body energy
      int iD1;                          //index into set of backbone defs
      double& D1e(int iD1){
         return o_D1e.at(iD1);
      }
      tZ2(){}
   };
   class tZ2Z2 { /*pairs of deformable segments*/
   public:
      int oD1;                          //size of 2nd index
      std::vector<double> o_D1D1e;      //lattice 2-body energy
      std::vector<double> o_D1z;        //low bound over defs of 2nd seg
      double z;                         //low bound over defs of 1st and 2nd seg
      double& D1D1e(int iD1,int jD1){
         return o_D1D1e.at( iD1*oD1 +jD1);
      }
      double& D1z(int iD1){
         return o_D1z.at(iD1);
      }
      tZ2Z2(){}
   };
   class tY3 { /*for subset of degs of free M3, set of residues contributing
                 sc degs of free*/
   public:
      int R0;                           //residue index
      std::string aa;                   //residue name
      int C2a;                          //start index into amino acid rots
      int cC2;                          //num of rots
      int nQ6;                          //num of torsions variable within rots
      std::vector<double> o_C2e;        //lattice 1-body energy
      int iC2;                          //index into set of rots
      double& C2e(int iC2){
         return o_C2e.at(iC2);
      }
      tY3(){}
      void normalize();
   };
   class tY3Y3 { /*pairs of residues contributing side chain degrees of
                   freedom*/
   public:
      int oC2;                          //size of 2nd index
      std::vector<double> o_C2C2e;      //lattice 2-body energy
      std::vector<double> o_C2z;        //low bound over rots of 2nd sc
      double z;                         //low bound over rots of 1st and 2nd sc
      double& C2C2e(int iC2,int jC2){
         return o_C2C2e.at( iC2*oC2 +jC2);
      }
      double& C2z(int iC2){
         return o_C2z.at(iC2);
      }
      tY3Y3(){}
      void normalize();
   };
   class tD3 { /*side chain conformations*/
   public:
      std::vector<int> o_Y3iC2;         //initial lattice conf
      std::vector<double> o_U2chi;      //(degree) off lattice conf
      double f[10];                     //restch energy and components
      int& Y3iC2(int i){
         return o_Y3iC2.at( i);
      }
      double& U2chi(int i){
         return o_U2chi.at( i);
      }
      const double& U2chi(int i) const {
         return o_U2chi.at( i);
      }
      tD3(){}
      void operator=(tD3& a);
      void swap(tD3& a);
   };

private:
   const int m;                 //random number generator
   const int a;                 //
   const int q;                 //
   const int r;                 //
   int Tz[32];                  //
   const int nT;                //
   const int d;                 //
   int SEED;                    //
   int iT;                      //

public:
   std::vector<tZ0> Z0;         //chains
   std::vector<tQ1> Q1;         //torsions [forward order]
   std::vector<tF1> F1;         //atoms [backward order]
   std::vector<tG1> G1;         //atoms in chain forward groups
   std::vector<tQ2> Q2;         //torsions contracted [forward order]
   std::vector<tGEN> GEN;       //generated groups
   std::vector<tG6> G6;         //generated residues used in screen
   bool LARGE_SUBSPACE;         //bb=2 for >15 search subspace residues
   double w;                    //weight of restch in screen energy
   std::vector<tZ2> Z2;         //deformable segments
   int oZ2;                     //
   std::vector<tZ2Z2> o_Z2Z2;   //pairs of deformable segments
   int nD2;                     //number
   int iD2;                     //index
   std::vector<tD2> D2;         //combined backbone deformations
   int oY3;                     //number
   std::vector<tY3> Y3;         //side chains in a packing unit
   std::vector<tY3Y3> o_Y3Y3;   //pairs of side chains
   std::vector<int> o_Y3Q6Q1;   //mapping to index of forward torsion
   bool CLASH;                  //confs contain multiple clashes
   int nD3;                     //number
   int iD3;                     //index
   std::vector<tD3> D3;         //side chain confs

   tZ2Z2& Z2Z2(int iZ2,int jZ2){
      return o_Z2Z2.at( iZ2*oZ2 +jZ2 -((iZ2+1)*(iZ2+2))/2);
   }
   tY3Y3& Y3Y3(int iY3,int jY3){
      return o_Y3Y3.at( iY3*oY3 +jY3 -((iY3+1)*(iY3+2))/2);
   }
   int& Y3Q6Q1(int i,int j){
      return o_Y3Q6Q1.at( i*5 +j);
   }
   const int& Y3Q6Q1(int i,int j) const {
      return o_Y3Q6Q1.at( i*5 +j);
   }

   double random_gen(){
      double z= (double(Tz[iT])/double(m));
      int jT=(Tz[iT]/d);
      int k=SEED/q;
      SEED=a*(SEED-k*q)-r*k;
      if( SEED<0 )SEED+=m;
      Tz[iT]=SEED;
      iT=jT;
      return z;
   }
   void CLUSTER(const std::string& MODE,
                int oU2);
   void ORDER(const std::string& MODE);
   void INSERT(const std::string& MODE);

   Glo_Automatic(int oZ0,int oQ1,int oF1,int oQ2,int oD2,int oR1,int o,
                 int oU2,int oG6,int oD3):
      m(2147483647),
      a(16807),
      q(127773),
      r(2836),
      nT(32),
      d(m/nT+1),
      SEED(1111111111),
      Z0(oZ0),
      Q1(oQ1),
      F1(oF1),
      G1(oF1),
      Q2(oQ2),
      GEN(oQ1),
      G6(oG6),
      oZ2(o),
      D2( 2*oD2,tD2(oR1,oZ2,oU2)),
      D3(oD3)
   {
      for(int i=0;i<8;i++){
         int k=SEED/q;
         SEED=a*(SEED-k*q)-r*k;
         if( SEED<0 )SEED+=m;
      }
      for(int jT= 0;jT<nT;jT++){
         int k=SEED/q;
         SEED=a*(SEED-k*q)-r*k;
         if( SEED<0 )SEED+=m;
         Tz[jT]=SEED;
      }
      iT=(SEED/d);
   }
};

#endif
