import Foundation

class Structure {
   struct tZ0 { /*chain in system of molecules*/
      var R0a: Int              //start index into set of residues
      var cR0: Int              //number of residues
      init() {
         R0a = -1
         cR0 = -1
      }
   }
   struct tR0 { /*residue in chains*/
      var aa: String            //4 char residue name
      var L0: Int               //index into set of residue types
      var c1: Character         //1 char class name
      var cha: Character        //pdb chain identifier
      var res: Int              //pdb residue sequence number
      var ins: Character        //pdb insertion character
      var P1a: Int              //start index into set of atoms
      var cP1: Int              //number of atoms
      var Z0: Int               //index into set of chains
      init() {
         aa = ""
         L0 = -1
         c1 = "a"
         cha = " "
         res = -1
         ins = " "
         P1a = -1
         cP1 = -1
         Z0 = -1
      }
   }
   struct tP1 { /*physical atom in chains [iupac order]*/
      var atm: String           //4 char atom name
      var typ: Int              //atom type
      var sub: Int              //position coordinates flag, {unset,set}={0,1}
      var x: Coordinates        //atom coordinates (angstrom)
      init() {
         atm = ""
         typ = -1
         sub = 0
         x = Coordinates()
      }
   }
   struct tS0 { /*disulfide bond in system of molecules*/
      var iZ0: Int              //chain 1st CYS
      var iR0: Int              //res
      var jZ0: Int              //chain 2nd CYS
      var jR0: Int              //res
      init() {
         iZ0 = -1
         iR0 = -1
         jZ0 = -1
         jR0 = -1
      }
   }
   struct tD { /*H-bond donor*/
      var P1: Int               //index to atoms
      var u: Coordinates        //unit vector along D--H
      init() {
         P1 = -1
         u = Coordinates()
      }
   }
   struct tA { /*H-bond acceptor*/
      var P1: Int               //index to atoms
      init() {
         P1 = -1
      }
   }

   var eregPath: String         //path name to ereg directory
   var mol: String              //name of system of molecules
   var nZ0: Int = 0             //number of chains
   var Z0: Array<tZ0> = []      //set of chains
   var R0: Array<tR0> = []      //set of residues
   var P1: Array<tP1> = []      //set of atoms
   var nS0: Int = 0             //number of disulfide crosslinks
   var S0: Array<tS0> = []      //set of disulfide crosslinks

   func iatom(_ iR0: Int,_ atm: String) -> Int {
      let mP1: Int = R0[iR0].P1a
      let nP1: Int = (mP1 - 1 + R0[iR0].cP1)
      for jP1: Int in mP1...nP1 {
         if( P1[jP1].atm == atm ){
            return jP1
         }
      }
      return -1
   }

   init(residueMappings: ResidueMappings, rootPath: String, filename: String) {
      eregPath = rootPath
      mol = filename
      var i: String.Index
      var j: String.Index
      var k: String.Index
      var l: String.Index
//
//
// read pdb file
//
      let path: String = eregPath + mol + ".pdb"
      print(path)
      let pathURL: URL = URL(fileURLWithPath: path)
      var Lrec: Array<String> = []
      do {
         let data: Data = try Data(contentsOf: pathURL)
         let contents: String? = String(data: data, encoding: .utf8)
         if( contents == nil ) {
            print("ERROR: Failure reading file \(mol).pdb.")
            return
         }
         Lrec = contents!.components(separatedBy: "\n") as [String]
      } catch {
         print("ERROR: File \(mol).pdb not found.")
         return
      }
//
//
// iterate over lines of file
//
      Z0.append( tZ0())
      var iZ0: Int = 0
      Z0[iZ0].R0a = 0
      Z0[iZ0].cR0 = 0
      var iR0: Int = -1
      var jP1: Int = -1
      var grp: String = "          "
      var aa: String = ""
      var AMINOACID: Bool = false
      var NEWCHAIN: Bool = false
      for iL: Int in  0..<Lrec.count {
         let rec: String = Lrec[iL]
         if( rec == "" ){ continue }
         i = rec.startIndex
         j = rec.index(i, offsetBy:  3)
         if( rec[i..<j] == "TER" ){
            NEWCHAIN = true
            continue
         }
         j = rec.index(i, offsetBy:  6)
         if( rec[i..<j] != "ATOM  " ){ continue }
         let i_cha: String.Index = rec.index(i, offsetBy: 21)
         let i_ins: String.Index = rec.index(i, offsetBy: 26)
//
//
// process rec
//
         let x: Coordinates = Coordinates ()
         j = rec.index(i, offsetBy: 30)
         k = rec.index(j, offsetBy:  8)
         x.x.0 = Double( rec[j..<k].trim())!
         j = rec.index(i, offsetBy: 38)
         k = rec.index(j, offsetBy:  8)
         x.x.1 = Double( rec[j..<k].trim())!
         j = rec.index(i, offsetBy: 46)
         k = rec.index(j, offsetBy:  8)
         x.x.2 = Double( rec[j..<k].trim())!
         j = rec.index(i, offsetBy: 17)
         k = rec.index(j, offsetBy: 10)
         if( rec[j..<k] != grp ){
            grp = String( rec[j..<k])
            if( iR0 >= Z0[iZ0].R0a ){
               if( (rec[i_cha] != R0[iR0].cha) || NEWCHAIN ){
                  NEWCHAIN = false
                  Z0.append( tZ0())
                  iZ0 += 1
                  Z0[iZ0].R0a = (iR0 + 1)
                  Z0[iZ0].cR0 = 0
                  if( AMINOACID ){
                     j = aa.startIndex
                     k = aa.index(j, offsetBy:  3)
                     aa =  aa[j..<k] + "e"
                     R0[iR0].aa = aa
                     R0[iR0].L0 = residueMappings.iresidue(aa: aa)
                     if( R0[iR0].L0 == -1 ){
                        let err = "ERROR: Unmatched residue name in PDB file."
                                + " iZ0 = \(iZ0 + 1)"
                                + " iR0 = \(iR0 + 1)"
                                + " aa = \(aa).\n"
                        print( err)
                        exit( 1)
                     }
                     R0[iR0].cP1 += 1
                     jP1 += 1
                     P1[jP1].atm = " OXT"
                     P1[jP1].typ = 13
                  }
               }
            }
            R0.append( tR0())
            iR0 += 1
            Z0[iZ0].cR0 += 1
            j = rec.index(i, offsetBy: 17)
            k = rec.index(j, offsetBy:  4)
            aa = String( rec[j..<k])
            var iL0: Int = residueMappings.iresidue(aa: aa)
            if( iL0 == -1 ){
               let err = "ERROR: Unmatched residue name in PDB file."
                       + " iZ0 = \(iZ0 + 1)"
                       + " iR0 = \(iR0 + 1)"
                       + " aa = \(aa).\n"
               print( err)
               exit( 1)
            }
            AMINOACID = (residueMappings.L0[iL0].c1 == Character( "a"))
            if( (iR0 == Z0[iZ0].R0a) && AMINOACID ){
               j = aa.startIndex
               k = aa.index(j, offsetBy:  3)
               aa = "e" + aa[j..<k]
               iL0 = residueMappings.iresidue(aa: aa)
               if( iL0 == -1 ){
                  let err = "ERROR: Unmatched residue name in PDB file."
                          + " iZ0 = \(iZ0 + 1)"
                          + " iR0 = \(iR0 + 1)"
                          + " aa = \(aa).\n"
                  print( err)
                  exit( 1)
               }
            }
            R0[iR0].aa = aa
            R0[iR0].L0 = iL0
            R0[iR0].c1 = residueMappings.L0[iL0].c1
            if( P1.count > (jP1 + 1) ){
               P1.remove(at: jP1 + 1)
            }
            R0[iR0].P1a = (jP1 + 1)
            R0[iR0].cP1 = residueMappings.L0[iL0].cP0
            let mP0: Int = residueMappings.L0[iL0].P0a
            let nP0: Int = (mP0 - 1 + residueMappings.L0[iL0].cP0)
            var iP1: Int = jP1
            for iP0: Int in mP0...nP0 {
               P1.append( tP1())
               iP1 += 1
               P1[iP1].atm = residueMappings.P0[iP0].atm
               P1[iP1].typ = residueMappings.P0[iP0].typ
               P1[iP1].sub = 0
            }
            jP1 += R0[iR0].cP1
            if( AMINOACID ){
               P1.append( tP1())
               P1[jP1 + 1].sub = 0
            }
            R0[iR0].cha = rec[i_cha]
            j = rec.index(i, offsetBy: 22)
            k = rec.index(j, offsetBy:  4)
            R0[iR0].res = Int( rec[j..<k].trim())!
            R0[iR0].ins = rec[ i_ins]
            R0[iR0].Z0 = iZ0
         }
         j = rec.index(i, offsetBy: 12)
         k = rec.index(j, offsetBy:  4)
         var atm: String = String( rec[j..<k])
         let mP1: Int = R0[iR0].P1a
         let nP1: Int = (mP1 - 1 + R0[iR0].cP1)
         for iP1: Int in mP1...nP1 {
            if( P1[iP1].atm == atm ){
               P1[iP1].sub = 1
               P1[iP1].x = x
               atm = "xxxx"
               break
            }
         }
         if( atm != "xxxx" ){
            j = atm.index(atm.startIndex, offsetBy:  1)
            let buf: Character = atm[j]
            if      ( AMINOACID && (buf == Character( "O")) ){
               P1[nP1 + 1].sub = 1
               P1[nP1 + 1].x = x
            }else if( ((aa == "CYS ") || (aa == "eCYS")) && (atm == " HG ") ){
               P1.append( tP1())
               P1[nP1 + 1].atm=" HG "
               P1[nP1 + 1].typ = 3
               P1[nP1 + 1].sub = 1
               P1[nP1 + 1].x = x
               if      ( aa == "CYS " ) {
                  aa = "CYH "
               }else if( aa == "eCYS" ) {
                  aa = "eCYH"
               }
               R0[iR0].aa = aa
               R0[iR0].L0 = residueMappings.iresidue(aa: aa)
               if( R0[iR0].L0 == -1 ){
                  let err = "ERROR: Unmatched residue name in PDB file."
                          + " iZ0 = \(iZ0 + 1)"
                          + " iR0 = \(iR0 + 1)"
                          + " aa = \(aa).\n"
                  print( err)
                  exit( 1)
               }
               R0[iR0].cP1 += 1
               jP1 += 1
            }else{
               let err = "ERROR: Unmatched atom name in PDB file."
                       + " iZ0 = \(iZ0 + 1)"
                       + " iR0 = \(iR0 + 1)"
                       + " aa = \(aa)"
                       + " atm = \(atm).\n"
               print( err)
               exit( 1)
            }
         }
      }
      nZ0 = (iZ0 + 1)
      if( AMINOACID ){
         i = aa.startIndex
         j = aa.index(i, offsetBy:  3)
         aa =  aa[i..<j] + "e"
         R0[iR0].aa = aa
         R0[iR0].L0 = residueMappings.iresidue(aa: aa)
         if( R0[iR0].L0 == -1 ){
            let err = "ERROR: Unmatched residue name in PDB file."
                    + " iZ0 = \(iZ0 + 1)"
                    + " iR0 = \(iR0 + 1)"
                    + " aa = \(aa).\n"
            print( err)
            exit( 1)
         }
         R0[iR0].cP1 += 1
         jP1 += 1
         P1[jP1].atm = " OXT"
         P1[jP1].typ = 13
      }
      if( P1.count > (jP1 + 1) ){
         P1.remove(at: jP1 + 1)
      }
//
//
// disulfide bonds
//
      nS0 = 0
      for iZ0: Int in  0..<nZ0 {
         let mR0: Int = Z0[iZ0].R0a
         let nR0: Int = (mR0 - 1 + Z0[iZ0].cR0)
         for iR0: Int in mR0...nR0 {
            aa = R0[iR0].aa
            i = aa.index(after: aa.startIndex)
            j = aa.index(before: aa.endIndex)
            if( (aa[aa.startIndex..<j] != "CYS") &&
                (aa[i..<aa.endIndex] != "CYS") ){ continue }
            let iP1: Int = (R0[iR0].P1a - 1 + 10)
            if( P1[iP1].sub != 1 ){
               let err = "ERROR: Missing atom in PDB file."
                       + " iZ0 = \(iZ0 + 1)"
                       + " iR0 = \(iR0 + 1)"
                       + " aa = \(aa).\n"
               print( err)
               exit( 1)
            }
            let jZ0min: Int = iZ0
            let jZ0max: Int = (nZ0 - 1)
            for jZ0: Int in jZ0min...jZ0max {
               let jR0min: Int = ( jZ0 == iZ0 ) ? (iR0 + 1): Z0[jZ0].R0a
               let jR0max: Int = (Z0[jZ0].R0a - 1 + Z0[jZ0].cR0)
               for jR0: Int in jR0min...jR0max {
                  aa = R0[jR0].aa
                  if( (aa[aa.startIndex..<j] != "CYS") &&
                      (aa[i..<aa.endIndex] != "CYS") ){ continue }
                  let jP1: Int = (R0[jR0].P1a - 1 + 10)
                  if( P1[jP1].sub != 1 ){ continue }
                  let r: Double  = ( P1[jP1].x - P1[iP1].x).r()
                  if( r < (2.80) ){
                     S0.append( tS0())
                     S0[nS0].iZ0 = iZ0
                     S0[nS0].iR0 = iR0
                     S0[nS0].jZ0 = jZ0
                     S0[nS0].jR0 = jR0
                     nS0 += 1
                  }
               }
            }
         }
      }
//
//
// free cysteine
//
      for iZ0: Int in  0..<nZ0 {
         let mR0: Int = Z0[iZ0].R0a
         let nR0: Int = (mR0 - 1 + Z0[iZ0].cR0)
         for iR0: Int in mR0...nR0 {
            aa = R0[iR0].aa
            if( (aa != "CYS ") && (aa != "eCYS") && (aa != "CYSe") ){ continue }
            var FREECYS: Bool = true
            for iS0: Int in  0..<nS0 {
               if( (S0[iS0].iR0 == iR0) || (S0[iS0].jR0 == iR0) ) {
                  FREECYS = false
                  break
               }
            }
            if( FREECYS ){
               if      ( aa == "CYS " ) {
                  aa = "CYH "
               }else if( aa == "eCYS" ) {
                  aa = "eCYH"
               }else if( aa == "CYSe" ) {
                  aa = "CYHe"
               }
               R0[iR0].aa = aa
               R0[iR0].L0 = residueMappings.iresidue(aa: aa)
               if( R0[iR0].L0 == -1 ){
                  let err = "ERROR: Unmatched residue name in PDB file."
                          + " iZ0 = \(iZ0 + 1)"
                          + " iR0 = \(iR0 + 1)"
                          + " aa = \(aa).\n"
                  print( err)
                  exit( 1)
               }
               let iP1: Int = R0[iR0].P1a + 10
               P1.insert( tP1(), at: iP1)
               R0[iR0].cP1 += 1
               P1[iP1].atm=" HG "
               P1[iP1].typ = 3
               P1[iP1].sub = 0
               let jZ0min: Int = iZ0
               let jZ0max: Int = (nZ0 - 1)
               for jZ0: Int in jZ0min...jZ0max {
                  let jR0min: Int = ( jZ0 == iZ0 ) ? (iR0 + 1): Z0[jZ0].R0a
                  let jR0max: Int = (Z0[jZ0].R0a - 1 + Z0[jZ0].cR0)
                  for jR0: Int in jR0min...jR0max {
                     R0[jR0].P1a += 1
                  }
               }
            }
         }
      }
//
//
// center of mass
//
      var x: Coordinates = Coordinates()
      var n: Int = 0
      for iZ0: Int in  0..<nZ0 {
         let mR0: Int = Z0[iZ0].R0a
         let nR0: Int = (mR0 - 1 + Z0[iZ0].cR0)
         for iR0: Int in mR0...nR0 {
            var aa: String = R0[iR0].aa
            i = aa.startIndex
            j = aa.index(i, offsetBy:  1)
            k = aa.index(i, offsetBy:  2)
            l = aa.index(i, offsetBy:  3)
            if      ( aa[i]=="e" ){
               aa = aa[j...l] + " "
            }else if( aa[l]=="e" ){
               aa = aa[i...k] + " "
            }
            let mP1: Int = R0[iR0].P1a
            let nP1: Int = (mP1 - 1 + R0[iR0].cP1)
            for iP1: Int in mP1...nP1 {
               if( P1[iP1].sub == 0 ){ continue }
               let atm: String = P1[iP1].atm
               var iT2: Int = P1[iP1].typ
               if( (aa == "CYH ") && (atm == " HG ") ){ iT2 = 5 }
               if( iT2 ==  0 ){ continue }
               if( iT2 ==  3 ){ continue }
               n += 1
               x += P1[iP1].x
            }
         }
      }
      x /= Double( n)
//
//
// translate st origin is center of mass
//
      for iZ0: Int in  0..<nZ0 {
         let mR0: Int = Z0[iZ0].R0a
         let nR0: Int = (mR0 - 1 + Z0[iZ0].cR0)
         for iR0: Int in mR0...nR0 {
            let mP1: Int = R0[iR0].P1a
            let nP1: Int = (mP1 - 1 + R0[iR0].cP1)
            for iP1: Int in mP1...nP1 {
               if( P1[iP1].sub == 0 ){ continue }
               P1[iP1].x -= x
            }
         }
      }
   }

   func createobj(residueMappings: ResidueMappings) {
//    let PI: Double = 3.141592653589790
      let RAD: Double = 0.01745329251994328
      var i: String.Index
      var j: String.Index
      var k: String.Index
      var l: String.Index
//
//
// normalization factor
//
      let fac: Double = ( 0.125)
//
//
// material sizes of vertex and face arrays
//
      var Mo: Array<(Int,Int)> = Array(repeating: ( 0, 0), count: 8)
//
//
// bond segments
//
      for iZ0: Int in  0..<nZ0 {
         let mR0: Int = Z0[iZ0].R0a
         let nR0: Int = (mR0 - 1 + Z0[iZ0].cR0)
         for iR0: Int in mR0...nR0 {
            let iL0: Int = R0[iR0].L0
            let mP0: Int = residueMappings.L0[iL0].P0a
            let nP0: Int = (mP0 - 1 + residueMappings.L0[iL0].cP0)
            for iP0: Int in mP0...nP0 {
               let L: Bool = residueMappings.P0[iP0].L
               let B: Bool = residueMappings.P0[iP0].B
               let U: Bool = residueMappings.P0[iP0].U
               if( !L && !B && !U ){ continue }
               let iM: Int = residueMappings.P0[iP0].M
               if(  B ){
                  Mo[iM].0 += 20
                  Mo[iM].1 += 20
                  if(  U ){
                     Mo[iM].0 += 26
                     Mo[iM].1 += 40
                     if(  L ){
                        Mo[iM].0 += 26
                        Mo[iM].1 += 40
                     }
                  }
               }else{
                  if(  L && U ){
                     Mo[iM].0 += 42
                     Mo[iM].1 += 80
                  }
               }
            }
         }
      }
//
//
// ring closures
//
      for iZ0: Int in  0..<nZ0 {
         let mR0: Int = Z0[iZ0].R0a
         let nR0: Int = (mR0 - 1 + Z0[iZ0].cR0)
         for iR0: Int in mR0...nR0 {
            let c1: Character = R0[iR0].c1
            var aa: String = R0[iR0].aa
            if      ( c1 == Character("a") ){
               i = aa.startIndex
               j = aa.index(i, offsetBy:  1)
               k = aa.index(i, offsetBy:  2)
               l = aa.index(i, offsetBy:  3)
               if      ( aa[i] == "e" ){
                  aa = aa[j...l] + " "
               }else if( aa[l] == "e" ){
                  aa = aa[i...k] + " "
               }
               if      ( aa == "PHE " ){
                  Mo[ 1].0 += 20
                  Mo[ 1].1 += 20
               }else if( (aa == "HIS ") || (aa == "HIE ") || (aa == "HIP ") ){
                  Mo[ 1].0 += 20
                  Mo[ 1].1 += 20
               }else if( aa == "PRO " ){
                  Mo[ 1].0 += 20
                  Mo[ 1].1 += 20
               }else if( aa == "TRP " ){
                  Mo[ 1].0 += 40
                  Mo[ 1].1 += 40
               }else if( (aa == "TYR ") || (aa == "TYZ ") || (aa == "TYP ") ){
                  Mo[ 1].0 += 20
                  Mo[ 1].1 += 20
               }
            }else if( c1 == Character("r") ){
               Mo[ 0].0 += 20
               Mo[ 0].1 += 20
               if( (aa == "LNA ") || (aa == "CET ") ){
                  Mo[ 1].0 += 20
                  Mo[ 1].1 += 20
               }
            }else if( c1 == Character("b") ){
               if      ( (aa == "A   ") || (aa == "AP  ") || (aa == "G   ") || 
                         (aa == "GP  ") || (aa == "GM  ") ){
                  Mo[ 1].0 += 40
                  Mo[ 1].1 += 40
               }else if( (aa == "T   ") || (aa == "TM  ") || (aa == "C   ") || 
                         (aa == "CP  ") || (aa == "U   ") || (aa == "UM  ") ){
                  Mo[ 1].0 += 20
                  Mo[ 1].1 += 20
               }
            }
         }
      }
//
//
// disulfide bonds
//
      if( nS0 > 0 ){
         for _: Int in  0..<nS0 {
            Mo[ 5].0 += 20
            Mo[ 5].1 += 20
         }
      }
//
//
// H-bond connectivity
//
      var nD: Int = 0           //number of H-bond donors
      var D: Array<tD> = []     //set of H-bond donors
      var nA: Int = 0           //number of H-bond donors
      var A: Array<tA> = []     //set of H-bond donors
      for iZ0: Int in  0..<nZ0 {
         let mR0: Int = Z0[iZ0].R0a
         let nR0: Int = (mR0 - 1 + Z0[iZ0].cR0)
         for iR0: Int in mR0...nR0 {
            var aa: String = R0[iR0].aa
            i = aa.startIndex
            j = aa.index(i, offsetBy:  1)
            k = aa.index(i, offsetBy:  2)
            l = aa.index(i, offsetBy:  3)
            if      ( aa[i] == "e" ){
               aa = aa[j...l] + " "
            }else if( aa[l] == "e" ){
               aa = aa[i...k] + " "
            }
            let mP1: Int = R0[iR0].P1a
            let nP1: Int = (mP1 - 1 + R0[iR0].cP1)
            for iP1: Int in mP1...nP1 {
               if( P1[iP1].sub == 0 ){ continue }
               let atm: String = P1[iP1].atm
               var iT2: Int = P1[iP1].typ
               var rcut: Double = (1.11)
               if( (aa == "CYH ") && (atm == " HG ") ){
                  iT2 = 5
                  rcut = (1.35)
               }
               if( (iT2 ==  2) || (iT2 ==  5) ){
                  var u: Coordinates = Coordinates()
                  var kP1: Int = -1
                  for lP1: Int in mP1...nP1 {
                     if( P1[lP1].sub == 0 ){ continue }
                     if( lP1 == iP1 ){ continue }
                     u = ( P1[iP1].x - P1[lP1].x)
                     let r: Double  =  u.r()
                     if( r < rcut ){
                        u /= r
                        kP1 = lP1
                        break
                     }
                  }
                  if( kP1 == -1 ){
                     let err = "ERROR: Missing H-bond vector.\n"
                     print( err)
                     exit( 1)
                  }
                  D.append( tD())
                  D[nD].P1 = iP1
                  D[nD].u = u
                  nD = nD + 1
               }else if( (iT2 == 12) || (iT2 == 13) || (iT2 == 14) ||
                         (iT2 == 15) || (iT2 == 17) ){
                  A.append( tA())
                  A[nA].P1 = iP1
                  nA = nA + 1
               }
            }
         }
      }
      let C75: Double = cos( 75.00 * RAD)
      let a0 = ((0.80) - C75) / ((1.00) - C75)
      let a1 = (0.20) / ((1.00) - C75)
      for iD: Int in  0..<nD {
         let iP1: Int = D[iD].P1
         let u: Coordinates = D[iD].u
         for jA: Int in  0..<nA {
            let jP1: Int = A[jA].P1
            var v: Coordinates = ( P1[jP1].x  - P1[iP1].x)
            let r: Double = v.r()
            if( r > (2.75) ){ continue }
            v /= r
            let Cthet: Double = dot(u,v)
            if( Cthet < C75 ){ continue }
            let h = 2.75 * (a0 + a1 * Cthet)
            if( r > h ){ continue }
            Mo[ 7].0 += 168
            Mo[ 7].1 += 320
         }
      }
//
//
// initiate tube
//
      let tub: Tube = Tube(Mo: Mo)
//
//
// add bond segments
//
      for iZ0: Int in  0..<nZ0 {
         let mR0: Int = Z0[iZ0].R0a
         let nR0: Int = (mR0 - 1 + Z0[iZ0].cR0)
         for iR0: Int in mR0...nR0 {
            let iL0: Int = R0[iR0].L0
            let mP0: Int = residueMappings.L0[iL0].P0a
            let nP0: Int = (mP0 - 1 + residueMappings.L0[iL0].cP0)
            for iP0: Int in mP0...nP0 {
               let L: Bool = residueMappings.P0[iP0].L
               let B: Bool = residueMappings.P0[iP0].B
               let U: Bool = residueMappings.P0[iP0].U
               if( !L && !B && !U ){ continue }
               let iP1: Int = iatom(iR0, residueMappings.P0[iP0].atm)
               let x0: Coordinates = P1[iP1].x
               var x1: Coordinates
               if(  B ){
                  let jR0: Int = (iR0 + residueMappings.P0[iP0].grp1)
                  let jP1: Int = iatom(jR0,residueMappings.P0[iP0].atm1)
                  x1 = P1[jP1].x
               }else{
                  x1 = x0
               }
               let iM: Int = residueMappings.P0[iP0].M
               tub.addbond(L,B,U,x0,x1,iM)
            }
         }
      }
//
//
// add ring closures
//
      do {
         let L: Bool = false
         let B: Bool = true
         let U: Bool = false
         for iZ0: Int in  0..<nZ0 {
            let mR0: Int = Z0[iZ0].R0a
            let nR0: Int = (mR0 - 1 + Z0[iZ0].cR0)
            for iR0: Int in mR0...nR0 {
               let c1: Character = R0[iR0].c1
               var aa: String = R0[iR0].aa
               if      ( c1 == Character("a") ){
                  let iM: Int =  1
                  i = aa.startIndex
                  j = aa.index(i, offsetBy:  1)
                  k = aa.index(i, offsetBy:  2)
                  l = aa.index(i, offsetBy:  3)
                  if      ( aa[i] == "e" ){
                     aa = aa[j...l] + " "
                  }else if( aa[l] == "e" ){
                     aa = aa[i...k] + " "
                  }
                  if      ( aa == "PHE " ){
                     let iP1: Int = iatom(iR0," CZ ")
                     let jP1: Int = iatom(iR0," CE2")
                     let x0: Coordinates = P1[iP1].x
                     let x1: Coordinates = P1[jP1].x
                     tub.addbond(L,B,U,x0,x1,iM)
                  }else if( (aa == "HIS ") || (aa == "HIE ") || (aa == "HIP ") ){
                     let iP1: Int = iatom(iR0," NE2")
                     let jP1: Int = iatom(iR0," CE1")
                     let x0: Coordinates = P1[iP1].x
                     let x1: Coordinates = P1[jP1].x
                     tub.addbond(L,B,U,x0,x1,iM)
                  }else if( aa == "PRO " ){
                     let iP1: Int = iatom(iR0," N  ")
                     let jP1: Int = iatom(iR0," CD ")
                     let x0: Coordinates = P1[iP1].x
                     let x1: Coordinates = P1[jP1].x
                     tub.addbond(L,B,U,x0,x1,iM)
                  }else if( aa == "TRP " ){
                     var iP1: Int = iatom(iR0," NE1")
                     var jP1: Int = iatom(iR0," CE2")
                     var x0: Coordinates = P1[iP1].x
                     var x1: Coordinates = P1[jP1].x
                     tub.addbond(L,B,U,x0,x1,iM)
                     iP1 = iatom(iR0," CZ3")
                     jP1 = iatom(iR0," CH2")
                     x0 = P1[iP1].x
                     x1 = P1[jP1].x
                     tub.addbond(L,B,U,x0,x1,iM)
                  }else if( (aa == "TYR ") || (aa == "TYZ ") || (aa == "TYP ") ){
                     let iP1: Int = iatom(iR0," CZ ")
                     let jP1: Int = iatom(iR0," CE2")
                     let x0: Coordinates = P1[iP1].x
                     let x1: Coordinates = P1[jP1].x
                     tub.addbond(L,B,U,x0,x1,iM)
                  }
//
               }else if( c1 == Character("r") ){
                  var iM: Int =  0
                  var iP1: Int = iatom(iR0," O4'")
                  var jP1: Int = iatom(iR0," C1'")
                  var x0: Coordinates = P1[iP1].x
                  var x1: Coordinates = P1[jP1].x
                  tub.addbond(L,B,U,x0,x1,iM)
                  if( (aa == "LNA ") || (aa == "CET ") ){
                     iM =  1
                     iP1 = iatom(iR0," C4'")
                     jP1 = iatom(iR0," C6'")
                     x0 = P1[iP1].x
                     x1 = P1[jP1].x
                     tub.addbond(L,B,U,x0,x1,iM)
                  }
//
               }else if( c1 == Character("b") ){
                  let iM: Int =  1
                  if      ( (aa == "A   ") || (aa == "AP  ") || (aa == "G   ") || 
                            (aa == "GP  ") || (aa == "GM  ") ){
                     var iP1: Int = iatom(iR0," C5 ")
                     var jP1: Int = iatom(iR0," N7 ")
                     var x0: Coordinates = P1[iP1].x
                     var x1: Coordinates = P1[jP1].x
                     tub.addbond(L,B,U,x0,x1,iM)
                     iP1 = iatom(iR0," C6 ")
                     jP1 = iatom(iR0," N1 ")
                     x0 = P1[iP1].x
                     x1 = P1[jP1].x
                     tub.addbond(L,B,U,x0,x1,iM)
                  }else if( (aa == "T   ") || (aa == "TM  ") || (aa == "C   ") || 
                            (aa == "CP  ") || (aa == "U   ") || (aa == "UM  ") ){
                     let iP1: Int = iatom(iR0," C5 ")
                     let jP1: Int = iatom(iR0," C4 ")
                     let x0: Coordinates = P1[iP1].x
                     let x1: Coordinates = P1[jP1].x
                     tub.addbond(L,B,U,x0,x1,iM)
                  }
//
               }
            }
         }
      }
//
//
// add disulfide bonds
//
      if( nS0 > 0 ){
         let L: Bool = false
         let B: Bool = true
         let U: Bool = false
         let iM: Int =  5
         for iS0: Int in  0..<nS0 {
            let iR0: Int = S0[iS0].iR0
            let jR0: Int = S0[iS0].jR0
            let iP1: Int = iatom(iR0," SG ")
            let jP1: Int = iatom(jR0," SG ")
            let x0: Coordinates = P1[iP1].x
            let x1: Coordinates = P1[jP1].x
            tub.addbond(L,B,U,x0,x1,iM)
         }
      }
//
//
// map H-bond connectivity
//
      do {
         let L: Bool = true
         let B: Bool = false
         let U: Bool = true
         var x0: Coordinates
         let iM: Int =  7
         for iD: Int in  0..<nD {
            let iP1: Int = D[iD].P1
            let u: Coordinates = D[iD].u
            for jA: Int in  0..<nA {
               let jP1: Int = A[jA].P1
               var v: Coordinates = ( P1[jP1].x  - P1[iP1].x)
               let r: Double = v.r()
               if( r > (2.75) ){ continue }
               v /= r
               let Cthet: Double = dot(u,v)
               if( Cthet < C75 ){ continue }
               let h = 2.75 * (a0 + a1 * Cthet)
               if( r > h ){ continue }
               var z: Double = (  0.000)
               for _: Int in 1...4 {
                  z += (  0.200)
                  x0 = ( z * P1[iP1].x  + ( (1.00)  - z) * P1[jP1].x)
                  tub.addbond(L,B,U,x0,x0,iM)
               }
            }
         }
      }
//
//
// output .obj file
//
      do {
         var contents: String = ""
         contents += "# OBJ File: \(mol)\n"
         contents += "mtllib \(mol).mtl\n"
         var m: Int = 1
         for iM: Int in  0..<8 {
            contents += "g _\(iM)\n"
            let oV: Int = tub.M[iM].oV
            for jV: Int in  0..<oV {
               contents += "v"
                   + " \(String(format: "%.6f", fac * tub.M[iM].V[jV].x.0))"
                   + " \(String(format: "%.6f", fac * tub.M[iM].V[jV].x.1))"
                   + " \(String(format: "%.6f",-fac * tub.M[iM].V[jV].x.2))"
                   + "\n"
            }
            for jV: Int in  0..<oV {
               contents += "vn"
                   + " \(String(format: "%.6f", tub.M[iM].V[jV].c.0))"
                   + " \(String(format: "%.6f", tub.M[iM].V[jV].c.1))"
                   + " \(String(format: "%.6f",-tub.M[iM].V[jV].c.2))"
                   + "\n"
            }
            contents += "usemtl material_\(iM)\n"
            let oF: Int = tub.M[iM].oF
            for jF: Int in  0..<oF {
               contents += "f"
                   + "  \( m + tub.M[iM].F[jF].v.0)//\( m + tub.M[iM].F[jF].v.0)"
                   + "  \( m + tub.M[iM].F[jF].v.1)//\( m + tub.M[iM].F[jF].v.1)"
                   + "  \( m + tub.M[iM].F[jF].v.2)//\( m + tub.M[iM].F[jF].v.2)"
                   + "\n"
            }
            contents += "s off\n"
            m += oV
         }
         let path: String = eregPath + mol + ".obj"
//         print(path)
         let objURL: URL = URL(fileURLWithPath: path)
         if let data: Data = contents.data(using: .utf8) {
            do {
               try data.write(to: objURL)
               print("Success writing file \(mol).obj.")
            } catch {
               print("ERROR: Failure writing file \(mol).obj.")
            }
         }
      }
//
//
// output .mtl file
//
      do {
         var contents: String = ""
         contents += "# MTL File: \(mol)\n"
         contents += "# Material Count: 8\n"
         for iM: Int in  0..<8 {
            contents += "\n"
            contents += "newmtl material_\(iM)\n"
            contents += "Ns 96.0784\n"
            contents += "Ka   0.000   0.000   0.000\n"
            contents += "Kd"
                   + "  \( tub.M[iM].rgb.0)"
                   + "  \( tub.M[iM].rgb.1)"
                   + "  \( tub.M[iM].rgb.2)"
                   + "\n"
            contents += "Ks   0.000   0.000   0.000\n"
            contents += "Ke   0.000   0.000   0.000\n"
            contents += "Ni\( 1.00)\n"
            contents += "d \( 1.00)\n"
            contents += "illum 2\n"
         }
         let path: String = eregPath + mol + ".mtl"
//         print(path)
         let mtlURL: URL = URL(fileURLWithPath: path)
         if let data: Data = contents.data(using: .utf8) {
            do {
               try data.write(to: mtlURL)
               print("Success writing file \(mol).mtl.")
            } catch {
               print("ERROR: Failure writing file \(mol).mtl.")
            }
         }
      }
      return
   }

   func reslab() -> Array<ResLab> {
      let fac: Double = ( 0.125)
      var Llabel: Array<ResLab> = Array(repeating: ResLab(), count: R0.count)
      for iZ0: Int in  0..<nZ0 {
         let mR0: Int = Z0[iZ0].R0a
         let nR0: Int = (mR0 - 1 + Z0[iZ0].cR0)
         for iR0: Int in mR0...nR0 {
            let c1: Character = R0[iR0].c1
            let aa: String = R0[iR0].aa
            Llabel[iR0].rec = aa + String(iR0)
            var atm: String = "    "
            if      ( c1 == "a" ){
               atm = " CA "
            }else if( c1 == "e" ){
               if     ( aa == "NH2 " ){ atm = " N  " }
               else                   { atm = " CH3" }
            }else if( c1 == "s" ){
               atm = " O  "
            }else if( c1 == "r" ){
               atm = " C4'"
            }else if( c1 == "p" ){
               if     ( aa == "5OH " ){ atm = " O5'" }
               else if( aa == "3OH " ){ atm = " O3'" }
               else                   { atm = " P  " }
            }else if( c1 == "b" ){
               if     ( aa == "A   " ){ atm = " N9 " }
               else if( aa == "AP  " ){ atm = " N9 " }
               else if( aa == "G   " ){ atm = " N9 " }
               else if( aa == "GP  " ){ atm = " N9 " }
               else if( aa == "GM  " ){ atm = " N9 " }
               else                   { atm = " N1 " }
            }
            let mP1: Int = R0[iR0].P1a
            let nP1: Int = (mP1 - 1 + R0[iR0].cP1)
            for iP1: Int in mP1...nP1 {
               if( P1[iP1].sub == 0 ){ continue }
               if( P1[iP1].atm == atm ){
                  let x: (Double,Double,Double) = (fac * P1[iP1].x).x
                  Llabel[iR0].x = Float( x.0)
                  Llabel[iR0].y = Float( x.1)
                  Llabel[iR0].z = Float(-x.2)
                  break
               }
            }
         }
      }
      return Llabel
   }

   func atmlab() -> Array<AtmLab> {
      let fac: Double = ( 0.125)
      var Llabel: Array<AtmLab> = []
      for iZ0: Int in  0..<nZ0 {
         let mR0: Int = Z0[iZ0].R0a
         let nR0: Int = (mR0 - 1 + Z0[iZ0].cR0)
         for iR0: Int in mR0...nR0 {
            let mP1: Int = R0[iR0].P1a
            let nP1: Int = (mP1 - 1 + R0[iR0].cP1)
            for iP1: Int in mP1...nP1 {
               if( P1[iP1].sub == 0 ){ continue }
               var label: AtmLab = AtmLab()
               label.rec = String(iP1)
               let atm: String = P1[iP1].atm
               label.atm = atm
               let x: (Double,Double,Double) = (fac * P1[iP1].x).x
               label.x = Float( x.0)
               label.y = Float( x.1)
               label.z = Float(-x.2)
               Llabel.append( label)
            }
         }
      }
      return Llabel
   }
}
