import Foundation

class BaseScene {

  var sceneSize: CGSize
  var camera: Camera = Camera()
  let rootNode = Node()
  var renderables: [Renderable] = []
  var uniforms = Uniforms()
  let inputController = InputController()

  init(sceneSize: CGSize) {
    self.sceneSize = sceneSize
    setupScene()
    sceneSizeWillChange(to: sceneSize)
  }

  private func updatePlayer(deltaTime: Float){
    inputController.updatePlayer(deltaTime: deltaTime)
  }

  final func update(deltaTime: Float) {
    updatePlayer(deltaTime: deltaTime)
    uniforms.projectionMatrix = camera.projectionMatrix
    uniforms.viewMatrix = camera.viewMatrix
    updateScene(deltaTime: deltaTime)
    update(nodes: rootNode.children, deltaTime: deltaTime)
  }

  private func update(nodes: [Node], deltaTime: Float) {
    nodes.forEach { node in
      node.update(deltaTime: deltaTime)
      update(nodes: node.children, deltaTime: deltaTime)
    }
  }

  func updateScene(deltaTime: Float) {
    // override this to update your scene
  }

  final func add(node: Node, parent: Node? = nil, render: Bool = true) {
    if let parent = parent {
      parent.add(childNode: node)
    } else {
      rootNode.add(childNode: node)
    }
//    guard render == true,
//      let renderable = node as? Renderable
//    else { return }
//    renderables.append(renderable)
  }

  final func remove(node: Node) {
    if let parent = node.parent {
      parent.remove(childNode: node)
    } else {
      for child in node.children {
        child.parent = nil
      }
      node.children = []
    }
//    guard
//      node is Renderable,
//      let index = (renderables.firstIndex { $0 as? Node === node })
//    else { return }
//    renderables.remove(at: index)
  }

  func setupScene() {
    // override this to add objects to the scene
  }

  func sceneSizeWillChange(to size: CGSize) {
    camera.aspect = Float(size.width / size.height)
    sceneSize = size
  }

}
