import MetalKit

class Renderer: NSObject, ObservableObject {
  static var device: MTLDevice!
  static var commandQueue: MTLCommandQueue!
  static var library: MTLLibrary?
  static var colorPixelFormat: MTLPixelFormat!
  static var depthStencilState: MTLDepthStencilState!
  lazy var lights: [Light] = {
    return lighting()
  }()
  var scene: BaseScene?

  override init() {
    guard
      let device = MTLCreateSystemDefaultDevice()
    else {
      print("ERROR: Failure to set GPU device.")
      exit( 1)
    }
    Renderer.device = device
    Renderer.commandQueue = device.makeCommandQueue()!
    Renderer.library = device.makeDefaultLibrary()
    super.init()
    buildDepthStencilState()
  }

  func buildDepthStencilState() {
    let descriptor = MTLDepthStencilDescriptor()
    descriptor.depthCompareFunction = .less
    descriptor.isDepthWriteEnabled = true
    Renderer.depthStencilState = Renderer.device.makeDepthStencilState(descriptor: descriptor)
  }

}
extension Renderer: MTKViewDelegate {
  func mtkView(_ view: MTKView, drawableSizeWillChange size: CGSize) {
    scene?.sceneSizeWillChange(to: size)
  }

  func draw(in metalView: MTKView) {
    mtkView(metalView, drawableSizeWillChange: metalView.bounds.size)
    guard
      let commandBuffer = Renderer.commandQueue.makeCommandBuffer(),
      let descriptor = metalView.currentRenderPassDescriptor,
      let renderEncoder = commandBuffer.makeRenderCommandEncoder(descriptor: descriptor),
      let scene = scene
    else { return }
    let deltaTime = 1 / Float(metalView.preferredFramesPerSecond)
    scene.update(deltaTime: deltaTime)

    renderEncoder.setDepthStencilState(Renderer.depthStencilState)
    var fragmentUniforms = FragmentUniforms()
    fragmentUniforms.lightCount = UInt32(lights.count)
    renderEncoder.setFragmentBytes(&fragmentUniforms,
                                   length: MemoryLayout<FragmentUniforms>.stride,
                                   index: Int(BufferIndexFragmentUniforms.rawValue))
    renderEncoder.setFragmentBytes(&lights,
                                   length: MemoryLayout<Light>.stride * lights.count,
                                   index: Int(BufferIndexLights.rawValue))

    for renderable in scene.renderables {
      renderable.render(renderEncoder: renderEncoder,
                        uniforms: scene.uniforms)
    }
    renderEncoder.endEncoding()
    guard
      let drawable = metalView.currentDrawable
    else { return }
    commandBuffer.present(drawable)
    commandBuffer.commit()
  }

}
