import simd

func radians(fromDegrees degrees: Float) -> Float {
  return (degrees / 180) * Float.pi
}
func degrees(fromRadians radians: Float) -> Float {
  return (radians / Float.pi) * 180
}

struct Rectangle {
  var left: Float = 0
  var right: Float = 0
  var top: Float = 0
  var bottom: Float = 0
}

extension Float {
  var radiansToDegrees: Float {
    return (self / Float.pi) * 180
  }
  var degreesToRadians: Float {
    return (self / 180) * Float.pi
  }
}

extension float4x4 {
  init(translation: (Float,Float,Float)) {
    self = matrix_identity_float4x4
    columns.3.x = translation.0
    columns.3.y = translation.1
    columns.3.z = translation.2
  }

  init(scaling: (Float,Float,Float)) {
    self = matrix_identity_float4x4
    columns.0.x = scaling.0
    columns.1.y = scaling.1
    columns.2.z = scaling.2
  }

  init(scaling: Float) {
    self = matrix_identity_float4x4
    columns.3.w = 1 / scaling
  }

  init(euler: (Float,Float,Float)) {
    self = matrix_identity_float4x4
    let alp: Float = euler.0
    let bet: Float = euler.1
    let gam: Float = euler.2
    let Calp: Float = cos( alp)
    let Salp: Float = sin( alp)
    let Cbet: Float = cos( bet)
    let Sbet: Float = sin( bet)
    let Cgam: Float = cos( gam)
    let Sgam: Float = sin( gam)
    columns.0.x = Calp*Cbet*Cgam - Salp*Sgam
    columns.0.y = -Calp*Cbet*Sgam - Salp*Cgam
    columns.0.z = Calp*Sbet
    columns.1.x = Salp*Cbet*Cgam + Calp*Sgam
    columns.1.y = -Salp*Cbet*Sgam + Calp*Cgam
    columns.1.z = Salp*Sbet
    columns.2.x = -Sbet*Cgam
    columns.2.y = Sbet*Sgam
    columns.2.z = Cbet
  }

  static func identity() -> float4x4 {
    let matrix:float4x4 = matrix_identity_float4x4
    return matrix
  }

  func upperLeft() -> float3x3 {
    let x = columns.0.xyz
    let y = columns.1.xyz
    let z = columns.2.xyz
    return float3x3(columns: (x, y, z))
  }

  init(projectionFov fov: Float, near: Float, far: Float, aspect: Float, lhs: Bool = true) {
    let y = 1 / tan(fov * 0.5)
    let x = y / aspect
    let z = lhs ? far / (far - near) : far / (near - far)
    let X = SIMD4<Float>( x,  0,  0,  0)
    let Y = SIMD4<Float>( 0,  y,  0,  0)
    let Z = lhs ? SIMD4<Float>( 0,  0,  z, 1) : SIMD4<Float>( 0,  0,  z, -1)
    let W = lhs ? SIMD4<Float>( 0,  0,  z * -near,  0) : SIMD4<Float>( 0,  0,  z * near,  0)
    self.init()
    columns = (X, Y, Z, W)
  }

//// left-handed LookAt
//  init(eye: SIMD3<Float>, center: SIMD3<Float>, up: SIMD3<Float>) {
//    let z = normalize(eye - center)
//    let x = normalize(cross(up, z))
//    let y = cross(z, x)
//    let w = SIMD3<Float>(dot(x, -eye), dot(y, -eye), dot(z, -eye))
//
//    let X = float4(x.x, y.x, z.x, 0)
//    let Y = float4(x.y, y.y, z.y, 0)
//    let Z = float4(x.z, y.z, z.z, 0)
//    let W = float4(w.x, w.y, x.z, 1)
//    self.init()
//    columns = (X, Y, Z, W)
//  }

}

extension float3x3 {
  init(normalFrom4x4 matrix: float4x4) {
    self.init()
    columns = matrix.upperLeft().inverse.transpose.columns
  }
}

extension float4 {
  var xyz: SIMD3<Float> {
    get {
      return SIMD3<Float>(x, y, z)
    }
    set {
      x = newValue.x
      y = newValue.y
      z = newValue.z
    }
  }

  init(_ start: SIMD3<Float>, _ end: Float) {
    self.init(start.x, start.y, start.z, end)
  }
}
//
//
// euler angles
//
func update(x: (Float,Float,Float), d: (Float,Float,Float)) -> (Float,Float,Float) {
  let Q: Rotation_Matrix = Rotation_Matrix(euler: x)
  let P: Rotation_Matrix = Rotation_Matrix(euler: d)
  let R: Rotation_Matrix = P * Q
  return R.euler()
}
